% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/homophily.R
\name{homophily}
\alias{homophily}
\title{Estimate Network Formation Model}
\usage{
homophily(
  network,
  formula,
  data,
  fixed.effects = FALSE,
  init = list(),
  iteration = 1000,
  print = TRUE
)
}
\arguments{
\item{network}{matrix or list of sub-matrix of social interactions containing 0 and 1, where links are represented by 1}

\item{formula}{an object of class \link[stats]{formula}: a symbolic description of the model. The \code{formula} should be as for example \code{~ x1 + x2}
where \code{x1}, \code{x2} are explanatory variable of links formation}

\item{data}{an optional data frame, list or environment (or object coercible by \link[base]{as.data.frame} to a data frame) containing the variables
in the model. If not found in data, the variables are taken from \code{environment(formula)}, typically the environment from which \code{homophily} is called.}

\item{fixed.effects}{boolean indicating if sub-network heterogeneity as fixed effects should be included.}

\item{init}{(optional) list of starting values containing \code{beta}, an K-dimensional vector of the explanatory variables parameter,
\code{mu} an n-dimensional vector, and \code{nu} an n-dimensional vector, \code{smu2} the variance of \code{mu},
and \code{snu2} the variance of \code{nu},
where K is the number of explanatory variables and n is the number of individuals.}

\item{iteration}{the number of iterations to be performed.}

\item{print}{boolean indicating if the estimation progression should be printed.}
}
\value{
A list consisting of:
\item{n}{number of individuals in each network.}
\item{n.obs}{number of observations.}
\item{n.links}{number of links.}
\item{K}{number of explanatory variables.}
\item{posterior}{list of simulations from the posterior distribution.}
\item{iteration}{number of performed iterations.}
\item{init}{returned list of starting values.}
}
\description{
Estimate Network Formation Model
}
\examples{
\donttest{
set.seed(1234)
library(MASS)
M            <- 4 # Number of sub-groups
nvec         <- round(runif(M, 100, 500))
beta         <- c(.1, -.1)
Glist        <- list()
dX           <- matrix(0, 0, 2)
mu           <- list()
nu           <- list()
cst          <- runif(M, -1.5, 0)
smu2         <- 0.2
snu2         <- 0.2
rho          <- 0.8
Smunu        <- matrix(c(smu2, rho*sqrt(smu2*snu2), rho*sqrt(smu2*snu2), snu2), 2)
for (m in 1:M) {
  n          <- nvec[m]
  tmp        <- mvrnorm(n, c(0, 0), Smunu)
  mum        <- tmp[,1] - mean(tmp[,1])
  num        <- tmp[,2] - mean(tmp[,2])
  X1         <- rnorm(n, 0, 1)
  X2         <- rbinom(n, 1, 0.2)
  Z1         <- matrix(0, n, n)  
  Z2         <- matrix(0, n, n)
  
  for (i in 1:n) {
    for (j in 1:n) {
      Z1[i, j] <- abs(X1[i] - X1[j])
      Z2[i, j] <- 1*(X2[i] == X2[j])
    }
  }
  
  Gm           <- 1*((cst[m] + Z1*beta[1] + Z2*beta[2] +
                       kronecker(mum, t(num), "+") + rnorm(n^2)) > 0)
  diag(Gm)     <- 0
  diag(Z1)     <- NA
  diag(Z2)     <- NA
  Z1           <- Z1[!is.na(Z1)]
  Z2           <- Z2[!is.na(Z2)]
  
  dX           <- rbind(dX, cbind(Z1, Z2))
  Glist[[m]]   <- Gm
  mu[[m]]      <- mum
  nu[[m]]      <- num
}

mu  <- unlist(mu)
nu  <- unlist(nu)

out   <- homophily(network =  Glist, formula = ~ dX, fixed.effects = TRUE, 
                   iteration = 1e3)

# plot simulations
plot(out$posterior$beta[,1], type = "l")
abline(h = cst[1], col = "red")
plot(out$posterior$beta[,2], type = "l")
abline(h = cst[2], col = "red")
plot(out$posterior$beta[,3], type = "l")
abline(h = cst[3], col = "red")
plot(out$posterior$beta[,4], type = "l")
abline(h = cst[4], col = "red")

plot(out$posterior$beta[,5], type = "l")
abline(h = beta[1], col = "red")
plot(out$posterior$beta[,6], type = "l")
abline(h = beta[2], col = "red")

plot(out$posterior$sigma2_mu, type = "l")
abline(h = smu2, col = "red")
plot(out$posterior$sigma2_nu, type = "l")
abline(h = snu2, col = "red")
plot(out$posterior$rho, type = "l")
abline(h = rho, col = "red")

i <- 10
plot(out$posterior$mu[,i], type = "l")
abline(h = mu[i], col = "red")
plot(out$posterior$nu[,i], type = "l")
abline(h = nu[i], col = "red")
}
}
