% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plotFMM.R
\name{plotFMM}
\alias{plotFMM}
\title{Plot fitted FMM models}
\usage{
plotFMM(
  objFMM,
  components = FALSE,
  plotAlongPeriods = FALSE,
  use_ggplot2 = FALSE,
  legendInComponentsPlot = TRUE,
  textExtra = ""
)
}
\arguments{
\item{objFMM}{Object of class FMM}

\item{components}{A logical value indicating if the centered wave components of the model should be separately
plotted (case where it is \code{TRUE}). If \code{FALSE}, the default, the fitted FMM model
along with the observed data is plotted.}

\item{plotAlongPeriods}{A logical value indicating if more than one period should be plotted in the plots
by default. Its default value is \code{FALSE}.}

\item{use_ggplot2}{A logical value. If \code{FALSE}, the default, R base graphics are used. If \code{TRUE},
\code{'ggplot2'} library is used as graphics engine.}

\item{legendInComponentsPlot}{A logical value indicating whether the legend should be plotted in the components
plot. By defaults it is \code{TRUE}.}

\item{textExtra}{A character vector for extra text to be added to the titles of the plots.}
}
\value{
None if base R graphics are used, a named \code{ggplot2} list if \code{'ggplot2'} is used.
}
\description{
\code{plotFMM()} is used to plot fitted FMM models. The function can either
plot the fitted model against the data or each of the components of the model
separately. Optionally \code{'ggplot2'} can be used as graphic library.
}
\details{
{ \code{plotFMM()} can generate two types of plots: the basic plot compares the fitted model against the original data while the components plot represents separately the centered waves of the model (if the argument components is TRUE).

The function is also capable of plotting multiple periods if the data has more than one, as is the case in many applications such as chronobiology. In this case, the argument plotAlongPeriods should be TRUE. In the case of components plots the value taken by the latter argument is ignored as they are plotted along just one period.

While, by default, plots are created using base R graphics, 'ggplot2' can also be used for more aesthetic and customizable plots. Optional arguments legendInComponentsPlot and textExtra serve to control, respectively, whether a legend to the components plot should be added and adding extra text to the plot's title.
}
}
\examples{

# Simulates an scenario in which an FMM model is suitable,
res <- generateFMM(2,3,1.5,2.3,0.1,outvalues = TRUE,sigmaNoise = 0.3, plot=FALSE)
# then a FMM model is fitted to the data.
fit <- fitFMM(res$y, lengthAlphaGrid=20,lengthOmegaGrid=12)
plotFMM(fit)

# Components plot of FMM Model fitted to neuronal data with various optional aesthetics
data("neuronalSpike")
fittedFMM2<-fitFMM(neuronalSpike, nback=2,
                   lengthAlphaGrid = 24,lengthOmegaGrid = 10, numReps = 1)

plotFMM(fittedFMM2, components = TRUE)
plotFMM(fittedFMM2, components = TRUE,
        legendInComponentsPlot = FALSE,
        textExtra = "Neuronal Data")

# With ggplot2, customizable plots can be created,
library(ggplot2)
# standard plots
plotFMM(fittedFMM2, use_ggplot2 = TRUE)
# modify x-axis with original timePoints
timePoints <- getTimePoints(fittedFMM2)
nObs <- length(timePoints)
sTimePoints <- round(c(1, nObs*0.25, nObs*0.5, nObs*0.75, nObs))
plotFMM(fittedFMM2, use_ggplot2 = TRUE) +
  scale_x_continuous(breaks = sTimePoints,
                     labels = function(x) round(timePoints[x],2))
# and components plots
plotFMM(fittedFMM2, components = TRUE, use_ggplot2 = TRUE)

# Plot of fitted model to more than one period.
data("mouseGeneExp")
fittedFMM2<-fitFMM(mouseGeneExp, nPeriods = 2,
                   lengthAlphaGrid = 20,lengthOmegaGrid = 10)
plotFMM(fittedFMM2, plotAlongPeriods = TRUE)
}
