#' Multivariate Distance Matrix Regression
#'
#' \code{MDMR} allows a user to conduct multivariate distance matrix regression
#' using analytic p-values and measures of effect size described by McArtor et
#' al. (2017). Analytic p-values are computed using the R package CompQuadForm
#' (Duchesne & De Micheaux, 2010). It also facilitates the use of MDMR on
#' samples consisting of (hierarchically) clustered observations.
#'
#' @section Usage:
#'  To access this package's tutorial, type the following line into the console:
#'
#'  \code{vignette('mdmr-vignette')}
#'
#'  There are three primary functions that comprise this package:
#'  \code{\link{mdmr}}, which regresses a distance matrix onto a set of
#'  predictors, and \code{\link{delta}}, which computes measures of univariate
#'  effect size in the context of multivariate distance matrix regression. The
#'  third function \code{\link{mixed.mdmr}} facilitates the use of MDMR on
#'  (hierarchically) clustered samples using an approach analogous to the
#'  linearar mixed-effects model for univariate outcomes. The help files of all
#'  all three functions provide more general information than the package
#'  vignette.
#'
#' @references Davies, R. B. (1980). The Distribution of a Linear Combination of
#'  chi-square Random Variables. Journal of the Royal Statistical Society.
#'  Series C (Applied Statistics), 29(3), 323-333.
#'
#'  Duchesne, P., & De Micheaux, P.L. (2010). Computing the distribution of
#'  quadratic forms: Further comparisons between the Liu-Tang-Zhang
#'  approximation and exact methods. Computational Statistics and Data
#'  Analysis, 54(4), 858-862.
#'
#'  McArtor, D. B., Lubke, G. H., & Bergeman, C. S. (2017). Extending
#'  multivariate distance matrix regression with an effect size measure and the
#'  distribution of the test statistic. Psychometrika, 82, 1052-1077.
#'
#'  McArtor, D. B. (2017). Extending a distance-based approach to multivariate
#'  multiple regression (Doctoral Dissertation).
#'
#' @examples
#' ################################################################
#' ## Conducting MDMR on data comprised of independent observations
#' ################################################################
#' # Source data
#' data(mdmrdata)
#'
#' # Get distance matrix
#' D <- dist(Y.mdmr, method = 'euclidean')
#'
#' # Conduct MDMR
#' mdmr.res <- mdmr(X = X.mdmr, D = D)
#' summary(mdmr.res)
#'
#' ################################################################
#' ## Conducting MDMR on data comprised of dependent observations
#' ################################################################
#' # Source data
#' data("clustmdmrdata")
#'
#' # Get distance matrix
#' D <- dist(Y.clust)
#'
#' # Conduct mixed-MDMR
#' mixed.res <- mixed.mdmr(~ x1 + x2 + (x1 + x2 | grp),
#'                         data = X.clust, D = D)
#' summary(mixed.res)
#'
#' @importFrom CompQuadForm davies
#' @importFrom parallel mclapply
#' @importFrom lme4 lmer lmerControl
#' @importFrom car Anova
#'
#' @docType package
#' @name MDMR-package
#' @aliases MDMR
NULL
