% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/initiate.R
\name{init.quad}
\alias{init.quad}
\title{Q-dimensional grid of quadrature points.}
\usage{
init.quad(Q = 2, prior = list(mu = rep(0, Q), Sigma = diag(Q)),
  adapt = NULL, ip = 6, prune = FALSE, forcePD = FALSE, debug = FALSE)
}
\arguments{
\item{Q}{Number of dimensions. Defaults to 2. Only required when \code{mu} and \code{Sigma} are not provided.}

\item{prior}{List of prior mean \code{mu}, = \code{vector}, and covariance matrix \code{Sigma} = \code{matrix}, defaults to zero vector and identity matrix respectively.}

\item{adapt}{List of adaptive mean \code{mu}, = \code{vector}, and covariance matrix \code{Sigma} = \code{matrix}, if \code{NULL} no adaptation is used. Defaults to NULL.}

\item{ip}{Number of quadrature points \emph{per dimension}. Defaults to 6. Note that the total number of quadrature points is \code{ip^Q}.}

\item{prune}{Logical, should quadrature points with a very low weight be removed? Defaults to false. See details.}

\item{forcePD}{Logical, should \code{adapt} and \code{prior} arguments be forced to the neares positive definite matrix - if not already PD? If TRUE (Default: FALSE), \code{\link[Matrix]{nearPD}} is used to arrive at the closest PD matrix.}

\item{debug}{Logical, draws debugging plots when true.}
}
\value{
A list with a matrix \code{X} of \code{ip^Q} by \code{Q} quadrature points and a vector \code{W} of length \code{ip^Q} associated weights.
}
\description{
Creates a flattened, rotated grid that incorporates correlation through an eigenvalue decomposition of the covariance matrix.
}
\details{
Creates a Q-dimensional grid by calling \code{\link{expand.grid}} on Q vectors of unidimensional quadrature points obtained with \code{\link[fastGHQuad]{gaussHermiteData}}.
The grid is then corrected for a prior distribution, and can optionally be adapted around a previous estimate. The resultant grid can be pruned to remove quadrature points that are unlikely to add information.
}
\examples{
### basic quadrature grid /w pruning.
mu <- c(0,0)
sigma <- matrix(c(1,.5,.5,1),2,2)
grid <- init.quad(Q = 2, prior = list(mu = mu, Sigma = sigma), ip = 10, prune = FALSE)
grid2 <- init.quad(Q = 2, prior = list(mu = mu, Sigma = sigma), ip = 10, prune = TRUE)
library(mvtnorm)
normal <- rmvnorm(1000, mu, sigma)
# noise
plot(normal, xlim = c(-6,6), ylim = c(-6,6), pch = 19, col = rgb(0,0,0,.5))
# full quad grid
points(grid$X, cex = exp(grid$W)/max(exp(grid$W))*4, col = 'red', pch = 20)
# pruned quad grid
points(grid2$X, cex = exp(grid2$W)/max(exp(grid2$W))*4, col = 'green', pch = 20)


### Adaptive quadrature grid
prior <- list(mu = c(0,0), Sigma = matrix(c(1,.5,.5,1),2,2))
adapt <- list(mu = c(-2,2), Sigma = prior$Sigma / 2)
grid <- init.quad(Q = 2, prior, ip = 10, prune = FALSE)
library(mvtnorm)
normal <- rmvnorm(1000, adapt$mu, adapt$Sigma)
# noise, centered at (-2, 2)
plot(normal, xlim = c(-6,6), ylim = c(-6,6), pch = 19, col = rgb(0,0,0,.5))
# initial quad grid, centered at (0, 0)
points(grid$X, cex = exp(grid$W)/max(exp(grid$W))*4, col = 'red', pch = 20)
# adapted grid
grid2 <- init.quad(Q =2, prior, adapt = adapt, ip = 10, prune = TRUE)
points(grid2$X, cex = exp(grid2$W)/max(exp(grid2$W))*4, col = 'green', pch = 20)
# the grid is adapted to the latest estimate, but weighted towards the prior 
}
\seealso{
\code{\link[fastGHQuad]{gaussHermiteData}}, used to create unidimensional quadrature points, and \code{\link{eval.quad}} for evaluating the integral.
}

