% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MultivariateMoments.R
\name{ShrinkageMoments}
\alias{ShrinkageMoments}
\alias{M2.shrink}
\alias{M3.shrink}
\alias{M4.shrink}
\title{Functions for calculating shrinkage-based comoments of financial time series}
\usage{
M2.shrink(R, targets = 1, f = NULL)

M3.shrink(R, targets = 1, f = NULL, unbiasedMSE = FALSE, as.mat = TRUE)

M4.shrink(R, targets = 1, f = NULL, as.mat = TRUE)
}
\arguments{
\item{R}{an xts, vector, matrix, data frame, timeSeries or zoo object of
asset returns}

\item{targets}{vector of integers selecting the target matrices to shrink to. The first four
structures are, in order: 'independent marginals', 'independent and identical marginals', 
'observed 1-factor model' and 'constant correlation'. See Details.}

\item{f}{vector or matrix with observations of the factor, to be used with target 3. See Details.}

\item{unbiasedMSE}{TRUE/FALSE whether to use a correction to have an unbiased
estimator for the marginal skewness values, in case of targets 1 and/or 2, default FALSE}

\item{as.mat}{TRUE/FALSE whether to return the full moment matrix or only
the vector with the unique elements (the latter is advised for speed), default
TRUE}
}
\description{
calculates covariance, coskewness and cokurtosis matrices using linear shrinkage
between the sample estimator and a structured estimator
}
\details{
The coskewness and cokurtosis matrices are defined as the matrices of dimension 
p x p^2 and p x p^3 containing the third and fourth order central moments. They
are useful for measuring nonlinear dependence between different assets of the 
portfolio and computing modified VaR and modified ES of a portfolio.

Shrinkage estimation for the covariance matrix was popularized by Ledoit and 
Wolf (2003, 2004). An extension to coskewness and cokurtosis matrices by
Martellini and Ziemann (2010) uses the 1-factor and constant-correlation structured
comoment matrices as targets. In Boudt, Cornilly and Verdonck (2017) the framework
of single-target shrinkage for the coskewness and cokurtosis matrices is 
extended to a multi-target setting, making it possible to include several target matrices
at once. Also, an option to enhance small sample performance for coskewness estimation
was proposed, resulting in the option 'unbiasedMSE' present in the 'M3.shrink' function.

The first four target matrices of the 'M2.shrink', 'M3.shrink' and 'M4.shrink' 
correspond to the models 'independent marginals', 'independent and identical marginals', 
'observed 1-factor model' and 'constant correlation'. Coskewness shrinkage includes two
more options, target 5 corresponds to the latent 1-factor model proposed in Simaan (1993)
and target 6 is the coskewness matrix under central-symmetry, a matrix full of zeros.
For more details on the targets, we refer to Boudt, Cornilly and Verdonck (2017) and
the supplementary appendix.

If f is a matrix containing multiple factors, then the shrinkage estimator will
use each factor in a seperate single-factor model and use multi-target shrinkage
to all targets matrices at once.
}
\examples{

data(edhec)

# construct an underlying factor (market-factor, observed factor, PCA, ...)
f <- rowSums(edhec)

# multi-target shrinkage with targets 1, 3 and 4
# as.mat = F' would speed up calculations in higher dimensions
targets <- c(1, 3, 4)
sigma <- M2.shrink(edhec, targets, f)$M2sh
m3 <- M3.shrink(edhec, targets, f)$M3sh
m4 <- M4.shrink(edhec, targets, f)$M4sh

# compute equal-weighted portfolio modified ES
mu <- colMeans(edhec)
p <- length(mu)
ES(p = 0.95, portfolio_method = "component", weights = rep(1 / p, p), mu = mu, 
    sigma = sigma, m3 = m3, m4 = m4)

# compare to sample method
sigma <- cov(edhec)
m3 <- M3.MM(edhec)
m4 <- M4.MM(edhec)
ES(p = 0.95, portfolio_method = "component", weights = rep(1 / p, p), mu = mu, 
    sigma = sigma, m3 = m3, m4 = m4)

}
\references{
Boudt, Kris, Brian G. Peterson, and Christophe Croux. 2008.
Estimation and Decomposition of Downside Risk for Portfolios with Non-Normal
Returns. Journal of Risk. Winter.

Boudt, Kris, Cornilly, Dries and Verdonck, Tim. 2017. A Coskewness Shrinkage 
Approach for Estimating the Skewness of Linear Combinations of Random Variables. 
Submitted. Available at SSRN: https://ssrn.com/abstract=2839781

Ledoit, Olivier and Wolf, Michael. 2003. Improved estimation of the covariance matrix 
of stock returns with an application to portfolio selection. Journal of empirical 
finance, 10(5), 603-621.

Ledoit, Olivier and Wolf, Michael. 2004. A well-conditioned estimator for large-dimensional 
covariance matrices. Journal of multivariate analysis, 88(2), 365-411.

Martellini, Lionel and Ziemann, V\"olker. 2010. Improved estimates of higher-order 
comoments and implications for portfolio selection. Review of Financial 
Studies, 23(4), 1467-1502.

Simaan, Yusif. 1993. Portfolio selection and asset pricing: three-parameter framework. 
Management Science, 39(5), 68-577.
}
\seealso{
\code{\link{CoMoments}} \cr \code{\link{StructuredMoments}} \cr \code{\link{EWMAMoments}} \cr \code{\link{MCA}}
}
\author{
Dries Cornilly
}
\concept{co-moments}
\concept{moments}
