% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pifMatch.R
\name{pifMatch}
\alias{pifMatch}
\title{Pseudo-Invariant Features based Image Matching}
\usage{
pifMatch(
  img,
  ref,
  method = "cor",
  quantile = 0.95,
  returnPifMap = TRUE,
  returnSimMap = TRUE,
  returnModels = FALSE
)
}
\arguments{
\item{img}{RasterStack or RasterBrick. Image to be adjusted.}

\item{ref}{RasterStack or RasterBrick. Reference image.}

\item{method}{Method to calculate pixel similarity. Options: euclidean distance ('ed'), spectral angle ('sam') or pearson correlation coefficient ('cor').}

\item{quantile}{Numeric. Threshold quantile used to identify PIFs}

\item{returnPifMap}{Logical. Return a binary raster map ot pixels which were identified as pesudo-invariant features.}

\item{returnSimMap}{Logical. Return the similarity map as well}

\item{returnModels}{Logical. Return the linear models along with the adjusted image.}
}
\value{
Returns a List with the adjusted image and intermediate products (if requested). 
#' \itemize{
   \item \code{img}: the adjusted image
   \item \code{simMap}: pixel-wise similarity map (if \code{returnSimMap = TRUE})
   \item \code{pifMap}: binary map of pixels selected as pseudo-invariant features (if \code{returnPifMap = TRUE}) 
   \item \code{models}: list of linear models; one per layer (if \code{returnModels = TRUE})                          
}
}
\description{
Match one scene to another based on linear regression of pseudo-invariant features (PIF).
}
\details{
The function consists of three main steps:
First, it calculates pixel-wise similarity between the two rasters and identifies pseudo-invariant pixels based on 
a similarity threshold. 
In the second step the values of the pseudo-invariant pixels are regressed against each other in a linear model for each layer.
Finally the linear models are applied to all pixels in the \code{img}, thereby matching it to the reference scene.

Pixel-wise similarity can be calculated using one of three methods: euclidean distance (\code{method = "ed"}), spectral angle (\code{"sam"}) or pearsons correlation coefficient (\code{"cor"}).
The threshold is defined as a similarity quantile. Setting \code{quantile=0.95} will select all pixels with a similarity above the 95\% quantile as pseudo-invariant features.

Model fitting is performed with simple linear models (\code{\link[stats]{lm}}); fitting one model per layer.
}
\examples{
library(raster)

## Import Landsat example data
data(lsat)

## Create fake example data
## In practice this would be an image from another acquisition date
lsat_b <- log(lsat)  

## Run pifMatch and return similarity layer, invariant features mask and models
lsat_b_adj <- pifMatch(lsat_b, lsat, returnPifMap = TRUE, 
                         returnSimMap = TRUE, returnModels = TRUE)
\donttest{
## Pixelwise similarity
ggR(lsat_b_adj$simMap, geom_raster = TRUE)

## Pesudo invariant feature mask 
ggR(lsat_b_adj$pifMap)

## Histograms of changes
par(mfrow=c(1,3))
hist(lsat_b[[1]], main = "lsat_b")
hist(lsat[[1]], main = "reference")
hist(lsat_b_adj$img[[1]], main = "lsat_b adjusted")

## Model summary for first band
summary(lsat_b_adj$models[[1]])
}
}
