% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/EstMCEMspatial_USER.R
\name{MCEM.sclm}
\alias{MCEM.sclm}
\title{Censored Spatial Model Estimation via MCEM Algorithm}
\usage{
MCEM.sclm(y, x, cens, LI, LS, coords, init.phi, init.nugget,
  type = "exponential", kappa = 0, lower = c(0.01, 0.01), upper = c(30,
  30), MaxIter = 500, nMin = 20, nMax = 5000, error = 1e-05,
  show.SE = TRUE)
}
\arguments{
\item{y}{vector of responses.}

\item{x}{design matrix.}

\item{cens}{vector of censoring indicators. For each observation: \code{1} if censored/missing
and \code{0} otherwise.}

\item{LI}{lower limit of detection. For each observation: if non-censored \code{=y},
if left-censored/missing \code{=-Inf}, or \code{=LOD} if right/interval-censored.}

\item{LS}{upper limit of detection. For each observation: if non-censored \code{=y},
if right-censored/missing \code{=Inf}, or \code{=LOD} if left/interval-censored.}

\item{coords}{2D spatial coordinates.}

\item{init.phi}{initial value for the spatial scaling parameter.}

\item{init.nugget}{initial value for the nugget effect parameter.}

\item{type}{type of spatial correlation function: '\code{exponential}', '\code{gaussian}',
'\code{matern}', and '\code{pow.exp}' for exponential, gaussian, matern, and power exponential, respectively.}

\item{kappa}{parameter for all spatial correlation functions. See \code{\link{CovMat}}.}

\item{lower, upper}{vectors of lower and upper bounds for the optimization method. If unspecified, the default is
\code{c(0.01,0.01)} for lower and \code{c(30,30)} for upper.}

\item{MaxIter}{maximum number of iterations of the MCEM algorithm. By default \code{=500}.}

\item{nMin}{initial sample size for Monte Carlo integration. By default \code{=20}.}

\item{nMax}{maximum sample size for Monte Carlo integration. By default \code{=5000}.}

\item{error}{maximum convergence error. By default \code{=1e-5}.}

\item{show.SE}{\code{TRUE} or \code{FALSE}. It indicates if the standard errors should be estimated. By default \code{=TRUE}.}
}
\value{
The function returns an object of class \code{sclm} which is a list given by:

\item{Theta}{estimated parameters in all iterations, \eqn{\theta = (\beta, \sigma^2, \phi, \tau^2)}.}
\item{theta}{final estimation of \eqn{\theta = (\beta, \sigma^2, \phi, \tau^2)}.}
\item{beta}{estimated \eqn{\beta}.}
\item{sigma2}{estimated \eqn{\sigma^2}.}
\item{phi}{estimated \eqn{\phi}.}
\item{tau2}{estimated \eqn{\tau^2}.}
\item{EY}{MC approximation of the first moment for the truncated normal distribution.}
\item{EYY}{MC approximation of the second moment for the truncated normal distribution.}
\item{SE}{vector of standard errors of \eqn{\theta = (\beta, \sigma^2, \phi, \tau^2)}.}
\item{InfMat}{observed information matrix.}
\item{loglik}{log-likelihood for the MCEM method.}
\item{AIC}{Akaike information criterion.}
\item{BIC}{Bayesian information criterion.}
\item{Iterations}{number of iterations needed to converge.}
\item{ptime}{processing time.}
\item{range}{the effective range.}
}
\description{
This function returns the maximum likelihood (ML) estimates of the unknown parameters in Gaussian spatial models
with censored/missing responses via the MCEM algorithm. It supports left, right,
interval, or missing values in the dependent variable. It also computes the observed information
matrix using the method developed by \insertCite{louis1982finding;textual}{RcppCensSpatial}.
}
\details{
The spatial Gaussian model is given by

\eqn{Y = X\beta + \xi},

where \eqn{Y} is the \eqn{n x 1} vector of response, \eqn{X} is the \eqn{n x q} design matrix,
\eqn{\beta} is the \eqn{q x 1} vector of regression coefficients to be estimated, and \eqn{\xi}
is the error term which is normally distributed with zero-mean and covariance matrix
\eqn{\Sigma=\sigma^2 R(\phi) + \tau^2 I_n}. We assume that \eqn{\Sigma} is non-singular and
\eqn{X} has full rank \insertCite{diggle2007springer}{RcppCensSpatial}.

The estimation process was performed via the MCEM algorithm initially proposed by
\insertCite{wei1990monte;textual}{RcppCensSpatial}. The Monte Carlo integration starts with a
sample of size \code{nMin}; at each iteration, the sample size increases (\code{nMax-nMin})/\code{MaxIter},
and at the last iteration, the sample size is \code{nMax}. The random observations are sampled
through the slice sampling algorithm available in package \code{relliptical}.
}
\note{
The MCEM final estimates correspond to the mean of the estimates obtained at each iteration after deleting
the half and applying a thinning of 3.

To fit a regression model for non-censored data, just set \code{cens} as a vector of zeros.

Functions \code{print}, \code{summary}, and \code{plot} work for objects of class \code{sclm}.
}
\examples{
# Simulated example: censored and missing data
n = 50   # Test with another values for n
set.seed(1000)
coords = round(matrix(runif(2*n,0,15),n,2),5)
x = cbind(rnorm(n), rnorm(n))
data = rCensSp(c(2,-1),2,3,0.70,x,coords,"left",0.08,0,"matern",1)
data$yobs[20] = NA
data$cens[20] = 1; data$LI[20] = -Inf; data$LS[20] = Inf

fit = MCEM.sclm(y=data$yobs, x=data[,7:8], cens=data$cens, LI=data$LI,
             LS=data$LS, coords=data[,5:6], init.phi=2.50, init.nugget=0.75,
             type="matern", kappa=1, MaxIter=20, nMax=1000, error=1e-4)
print(fit)

\donttest{
# Application: TCDD concentration in Missouri
library(CensSpatial)
data("Missouri")
y = log(Missouri$V3)
cc = Missouri$V5
coord = cbind(Missouri$V1/100,Missouri$V2)
X = matrix(1,length(y),1)
LI = LS = y; LI[cc==1] = -Inf

fit2 = MCEM.sclm(y=y, x=X, cens=cc, LI=LI, LS=LS, coords=coord, init.phi=5,
              init.nugget=1, type="exponential", lower=c(1e-5,1e-5), upper=c(50,50),
              MaxIter=500, nMax=1000, error=1e-5)
summary(fit2)
plot(fit2)}
}
\references{
\insertAllCited
}
\seealso{
\code{\link{EM.sclm}}, \code{\link{SAEM.sclm}}, \code{\link{predict.sclm}}
}
\author{
Katherine L. Valeriano, Alejandro Ordonez, Christian E. Galarza and Larissa A. Matos.
}
