% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/custom.trend.R
\name{custom.trend}
\alias{custom.trend}
\title{Isolate trend component from mean sea level records via customised input
parameters and analysis}
\usage{
custom.trend(object, station_name = " ", iter = 10000, trend = c(1),
  DOF = " ", vlm = " ", plot = "TRUE", wdir = " ",
  save_summary = "FALSE")
}
\arguments{
\item{object}{an annual average mean sea level time series (refer \code{\link[stats]{ts}})
with NO missing data or an object of class "gap.fillview" (refer \code{\link{gap.fillview}})
with water levels (in millimetres).


\strong{Warning: } If input data files do not conform to these pre-conditions,
the analysis will be terminated. It should be further noted that the existence
of long period oscillations in global mean sea level have been well recognised
in the literature (eg. Chambers et al. (2012); Minobe (1999)). Therefore, in
order to be effective for climate change and sea level research, time series
input files are recommended to have a minimum length of at least 80 years
in order that the analysis can identify and isloate such signals. Time series
less than 80 years in length will be analysed but a warning will be displayed.
Time series less than 30 years are not permitted though.}

\item{station_name}{character string, providing the name of the data record.

\strong{Note: }This field can be left blank, however, it is retained for use
in banner labelling of graphical outputs.}

\item{iter}{numeric, enables a user defined number of iterations for
bootstrapping to determine error margins. The user range is [500 to 10000]
where 10000 is the default setting.\cr

\strong{Warning: }Although the default setting provides a more accurate basis
for estimating error margins, the degree of iterations slows the analysis and
can take several minutes to run.}

\item{trend}{numeric, enables the user to select the trend components
directly in the form of a single component or multiple components (eg.,
c(1) or c(1,2,3)). The default setting is c(1) as the first component will
always be trend, however, other components might also have trend characteristics
which can be diagnostically observed and optimised via the \code{\link{check.decomp}}
function.}

\item{DOF}{numeric, enables the user to optimise the degrees of freedom
for the fitted cubic smoothing spline to be applied to the trend. The default
setting is based on 1 degree of freedom every 8 years (Watson, 2018) and
this default is written to the console to enable the user to directly compare
with manually entered DOF settings. The DOF can be diagnostically observed and
optimised via the \code{\link{check.decomp}} function.}

\item{vlm}{numeric, enables a user defined quantum for vertical land motion
in mm/year within the range [-20 to 20]. This rate is used to convert the rate
of relative sea level rise to an estimate of geocentric sea level rise. Positive
rates of vlm are associated with land uplift, while conversely negative rates
of vlm are associated with subsidence. This can be left blank in which case
only estimates of relative mean sea level will be determined.}

\item{plot}{logical, if \dQuote{TRUE} then the original time series is
plotted to the screen along with the trend component and the result of gap
filling (where necessary). 95\% confidence intervals have also been applied.
Default = \dQuote{TRUE}.}

\item{wdir}{character string, providing the name of the directory to send
output files (e.g., \dQuote{C:/myproject/}) when the save_summary argument is set to \dQuote{TRUE}.
If this field is left blank the save_summary argument is switched off and a message
will be sent to the console.}

\item{save_summary}{logical, if \dQuote{TRUE} the object$Summary portion
of the returned value is exported direct to the defined  directory (wdir) and
saved as "detailed_summary_output.csv". Default = \dQuote{FALSE}.}
}
\value{
An object of class \dQuote{custom.trend} is returned with the following
elements:
\describe{
 \item{\strong{$Station.Name: }}{the name of the data record.}
 \item{\strong{$Summary: }}{a summary data frame of the relevant attributes
 relating to the trend and
  the inputted annual average data set, including:}
 \itemize{
  \item{$Year: input data; }
  \item{$MSL: input data; }
  \item{$Trend: mean sea level trend; }
  \item{$TrendSD: standard deviation of the determined mean sea level
  trend; }
  \item{$Vel: relative velocity (or first derivative) of mean sea level trend
  (mm/year); }
  \item{$VelSD: standard deviation of the velocity of the mean sea level
  trend; }
  \item{$Acc: acceleration (or second derivative) of mean sea level trend
  (mm/year/year); }
  \item{$AccSD: standard deviation of the acceleration of the mean sea level
  trend; }
  \item{$Resids: time series of uncorrelated residuals; and }
  \item{$FilledTS: gap-filled time series (where necessary). }
  \item{$VelGeo: geocentric velocity (or first derivative) of mean sea level
  trend (mm/year)(only where vertical land motion has been supplied). }
   }
}

\describe{
 \item{\strong{$Relative.Velocity: }}{outputs the peak relative velocity
 and the year in which it occurred.}
 \item{\strong{$Vertical.Land.Motion: }}{outputs the vertical land motion
 used to convert relative to geocentric velocity (user supplied input).}
 \item{\strong{$Geocentric.Velocity: }}{outputs the peak geocentric velocity
 and the year in which it occurred (if vertical land motion supplied).}
 \item{\strong{$Acceleration: }}{outputs the peak acceleration and the
 year in which it occurred.}
 \item{\strong{$Record.Length: }}{outputs details of the start, end and
 length of the input data set.}
 \item{\strong{$Fillgaps: }}{outputs the extent of missing data (years) in
 the original record and the gap filling method used (where necessary).}
 \item{\strong{$Bootstrapping.Iterations: }}{outputs the number of iterations
 used to generate the respective standard deviations for error margins.}
 \item{\strong{$Changepoints: }}{outputs the number and time at which
 changepoints in the variance of the uncorrelated residuals occur (if any).
 Where changepoints are identified, block bootstrapping procedures are used
 with residuals quarantined between changepoints.}
 \item{\strong{$Trend.Components: }}{outputs the components from the SSA
 decomposition of the original time series used to estimate the trend.}
 \item{\strong{$DOF.Fitted.Spline: }}{outputs the degrees of freedom used
 to fit the cubic smoothing spline to the trend in order to estimate velocity
 and acceleration.}
 }
}
\description{
Isolate trend component from mean sea level records via customised input
parameters and analysis
}
\details{
This function permits the customisation of key input parameters to
enable improved isolation of trend components (mean sea level) and estimated
associated velocities and accelerations. This function provides more flexibility
for the expert analyst than the \code{\link{msl.trend}} function which has fixed
inbuilt parameterisation based on the recommendations espoused in Watson (2018).
The selection of the "trend" and "DOF" parameters would be undertaken following
diagnostic analysis of the input time series via the \code{\link{check.decomp}}
function. The trend is isolated using Singular Spectrum Analysis, in particular,
aggregating components whose spectral energy in the low frequency bands exhibit
trend-like characteristics. Associated velocities and accelerations are
determined through the fitting of a cubic smoothing spline to the trend.
}
\examples{

data(Balt) # Baltimore mean sea level record
ts1 <- ts(Balt[2], start = Balt[1, 1]) # create time series input object
g <- gap.fillview(ts1, station_name = "Baltimore", fillgaps = 1) # SSA gap fill
\donttest{t <- custom.trend(g, station_name = "Baltimore (USA)", iter = 500, trend = c(1,2),
vlm = 0.6)}

data(t)
str(t) # check structure of object

}
\references{
Chambers, D.P., Merrifield, M.A., and Nerem, R.S., 2012. Is there
a 60 year oscillation in global mean sea level? \emph{Geophysical Research Letters}, 39(18).

Minobe, S., 1999. Resonance in bidecadal and pentadecadal climate oscillations
over the North Pacific: Role in climatic regime shifts. \emph{Geophysical Research Letters},
26(7), pp.855-858.

Watson, P.J., 2018. \emph{Improved Techniques to Estimate Mean Sea Level,
Velocity and Acceleration from Long Ocean Water Level Time Series to Augment
Sea Level (and Climate Change) Research.} PhD Thesis, University of New South
Wales, Sydney, Australia.
}
\seealso{
\code{\link{msl.trend}}, \code{\link{gap.fillview}}, \code{\link{check.decomp}},
\code{\link{t}}, \code{\link[stats]{ts}}, \code{\link{msl.fileplot}},
\code{\link{msl.screenplot}}, \code{\link{summary}}, \code{\link{Balt}}.
}
