% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fill_voxel_cloud.R
\name{filled_voxel_cloud}
\alias{filled_voxel_cloud}
\title{Produces a filled voxel cloud.}
\usage{
filled_voxel_cloud(data, res, d_clust, estimate_volume, message)
}
\arguments{
\item{data}{a data.frame or data.table containing the x, y, z, ... coordinates of a point cloud.}

\item{res}{numeric. Resolution of a voxel.}

\item{d_clust}{numeric. The distance to use for clustering, see the \link{distance_clustering} for more details.}

\item{estimate_volume}{logical. If TRUE the tree volume is computed as done in Vonderach et al. (2012).}

\item{message}{logical. If FALSE, messages are disabled. Default = TRUE.}
}
\value{
If \code{estimate_volume = FLASE} a data.frame or data.table containing the voxels coordinates is returned. If
        \code{estimate_volume = TRUE} a list containing the voxels coordinates and the estimated tree volume
        is returned.
}
\description{
This function produces a filled voxel cloud of a tree, i.e. a voxels cloud within which
             empty objects (e.g. trunk and large branches) are filled. The algorithm was inspired from
             the one described by Vonderach et al. (2012) with some modifications. First, the point cloud is
             is voxelized with a given (\code{res}) voxel resolution. The voxel cloud is then sliced into one voxel
             tick layers. Within a single layer different objects are then clustered based on their
             distance to each other (see the \link{distance_clustering} function for more details). Each
             cluster is then filled by addind voxels along the range of Y for each X value of the cluster and reversly along
             the range of X for each Y of the cluster. All unique voxels are then returned.
}
\examples{
#- import tls data
tls=data.table::fread(system.file("extdata", "Tree_t1.asc", package="VoxR"))

#- keep the tree trunk
tls=tls[z<=0]

#- run filled voxel voxelisation
filled = VoxR::filled_voxel_cloud(tls,0.02)

#- run usual voxelisation
voxels = VoxR::vox(tls,0.02)

#- compare filled voxel cloud to empty voxel cloud
VoxR::plot_voxels(filled,res = 0.02)
VoxR::plot_voxels(voxels,res = 0.02)

#- compare the volume estimate from Vonderach et al. 2012 to estimate based on voxel volume
#- run filled voxel voxelisation with volume estimation
filled = VoxR::filled_voxel_cloud(tls,0.01,estimate_volume = TRUE)

#- compare volumes
filled$estimated_volume # Vonderach
nrow(filled$filled_voxels)*0.01^3 # voxel based
}
\references{
Vonderach, C., Voegtle, T., & Adler, P. (2012). Voxel-based approach for estimating urban tree volume from
            terrestrial laser scanning data. International Archives of Photogrammetry, Remote Sensing and Spatial Information Sciences,
            39, 451-456.
}
