% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/uploadData.R
\name{bqr_upload_data}
\alias{bqr_upload_data}
\title{Upload data to BigQuery}
\usage{
bqr_upload_data(projectId = bqr_get_global_project(),
  datasetId = bqr_get_global_dataset(), tableId, upload_data,
  create = c("CREATE_IF_NEEDED", "CREATE_NEVER"),
  writeDisposition = c("WRITE_TRUNCATE", "WRITE_APPEND", "WRITE_EMPTY"),
  schema = NULL, sourceFormat = c("CSV", "DATASTORE_BACKUP",
  "NEWLINE_DELIMITED_JSON", "AVRO"), wait = TRUE, autodetect = FALSE,
  nullMarker = NULL, maxBadRecords = NULL, allowJaggedRows = FALSE,
  allowQuotedNewlines = FALSE, fieldDelimiter = NULL)
}
\arguments{
\item{projectId}{The BigQuery project ID.}

\item{datasetId}{A datasetId within projectId.}

\item{tableId}{ID of table where data will end up.}

\item{upload_data}{The data to upload, a data.frame object or a Google Cloud Storage URI}

\item{create}{Whether to create a new table if necessary, or error if it already exists.}

\item{writeDisposition}{How to add the data to a table.}

\item{schema}{If \code{upload_data} is a Google Cloud Storage URI, supply the data schema.  For \code{CSV} a helper function is available by using \link{schema_fields} on a data sample}

\item{sourceFormat}{If \code{upload_data} is a Google Cloud Storage URI, supply the data format.  Default is \code{CSV}}

\item{wait}{If uploading a data.frame, whether to wait for it to upload before returning}

\item{autodetect}{Experimental feature that auto-detects schema for CSV and JSON files}

\item{nullMarker}{Specifies a string that represents a null value in a CSV file. 
For example, if you specify \code{\\N}, BigQuery interprets \code{\\N} as a null value when loading a CSV file. The default value is the empty string.}

\item{maxBadRecords}{The maximum number of bad records that BigQuery can ignore when running the job}

\item{allowJaggedRows}{Whether to allow rows with variable length columns}

\item{allowQuotedNewlines}{Whether to allow datasets with quoted new lines}

\item{fieldDelimiter}{The separator for fields in a CSV file.  Default is comma - \code{,}}
}
\value{
TRUE if successful, FALSE if not.
}
\description{
Upload data to BigQuery
}
\details{
A temporary csv file is created when uploading from a local data.frame

For larger file sizes up to 5TB, upload to Google Cloud Storage first via \link[googleCloudStorageR]{gcs_upload} then supply the object URI of the form \code{gs://project-name/object-name} to the \code{upload_data} argument.  
  
You also need to supply a data schema.  Remember that the file should not have a header row.
}
\examples{

\dontrun{

 library(googleCloudStorageR)
 library(bigQueryR)
 
 gcs_global_bucket("your-project")
 
 ## custom upload function to ignore quotes and column headers
 f <- function(input, output) {
   write.table(input, sep = ",", col.names = FALSE, row.names = FALSE, 
               quote = FALSE, file = output, qmethod = "double")}
   
 ## upload files to Google Cloud Storage
 gcs_upload(mtcars, name = "mtcars_test1.csv", object_function = f)
 gcs_upload(mtcars, name = "mtcars_test2.csv", object_function = f)
 
 ## create the schema of the files you just uploaded
 user_schema <- schema_fields(mtcars)
 
 ## load files from Google Cloud Storage into BigQuery
 bqr_upload_data(projectId = "your-project", 
                datasetId = "test", 
                tableId = "from_gcs_mtcars", 
                upload_data = c("gs://your-project/mtcars_test1.csv", 
                                "gs://your-project/mtcars_test2.csv"),
                schema = user_schema)
 
 ## for big files, its helpful to create your schema on a small sample
 ## a quick way to do this on the command line is:
 # "head bigfile.csv > head_bigfile.csv"

## upload nested lists as JSON
the_list <- list(list(col1 = "yes", col2 = "no", 
                      col3 = list(nest1 = 1, nest2 = 3), col4 = "oh"),
                 list(col1 = "yes2", 
                      col2 = "n2o", col3 = list(nest1 = 5, nest2 = 7), 
                      col4 = "oh2"), 
                 list(col1 = "yes3", col2 = "no3", 
                      col3 = list(nest1 = 7, nest2 = 55), col4 = "oh3"))
   
bqr_upload_data(datasetId = "test", 
                tableId = "nested_list_json", 
                upload_data = the_list, 
                autodetect = TRUE)

}

}
\seealso{
url{https://cloud.google.com/bigquery/loading-data-post-request}
}
\concept{bigQuery upload functions}
