% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/operators.R
\name{laplacian}
\alias{laplacian}
\alias{\%laplacian\%}
\title{Numerical and Symbolic Laplacian}
\usage{
laplacian(
  f,
  var,
  params = list(),
  coordinates = "cartesian",
  accuracy = 4,
  stepsize = NULL,
  drop = TRUE
)

f \%laplacian\% var
}
\arguments{
\item{f}{array of \code{characters} or a \code{function} returning a \code{numeric} array.}

\item{var}{vector giving the variable names with respect to which the derivatives are to be computed and/or the point where the derivatives are to be evaluated. See \code{\link{derivative}}.}

\item{params}{\code{list} of additional parameters passed to \code{f}.}

\item{coordinates}{coordinate system to use. One of: \code{cartesian}, \code{polar}, \code{spherical}, \code{cylindrical}, \code{parabolic}, \code{parabolic-cylindrical} or a vector of scale factors for each varibale.}

\item{accuracy}{degree of accuracy for numerical derivatives.}

\item{stepsize}{finite differences stepsize for numerical derivatives. It is based on the precision of the machine by default.}

\item{drop}{if \code{TRUE}, return the Laplacian as a scalar and not as an \code{array} for scalar-valued functions.}
}
\value{
Scalar for scalar-valued functions when \code{drop=TRUE}, \code{array} otherwise.
}
\description{
Computes the numerical Laplacian of \code{functions} or the symbolic Laplacian of \code{characters} 
in arbitrary \href{https://en.wikipedia.org/wiki/Orthogonal_coordinates#Table_of_orthogonal_coordinates}{orthogonal coordinate systems}.
}
\details{
The Laplacian is a differential operator given by the divergence of the 
gradient of a scalar-valued function \eqn{F}, resulting in a scalar value giving 
the flux density of the gradient flow of a function. 
The \code{laplacian} is computed in arbitrary orthogonal coordinate systems using 
the scale factors \eqn{h_i}:

\deqn{\nabla^2F = \frac{1}{J}\sum_i\partial_i\Biggl(\frac{J}{h_i^2}\partial_iF\Biggl)}

where \eqn{J=\prod_ih_i}. When the function \eqn{F} is a tensor-valued function 
\eqn{F_{d_1\dots d_n}}, the \code{laplacian} is computed for each scalar component:

\deqn{(\nabla^2F)_{d_1\dots d_n} = \frac{1}{J}\sum_i\partial_i\Biggl(\frac{J}{h_i^2}\partial_iF_{d_1\dots d_n}\Biggl)}
}
\section{Functions}{
\itemize{
\item \code{\%laplacian\%}: binary operator with default parameters.
}}

\examples{
### symbolic Laplacian 
laplacian("x^3+y^3+z^3", var = c("x","y","z"))

### numerical Laplacian in (x=1, y=1, z=1)
f <- function(x, y, z) x^3+y^3+z^3
laplacian(f = f, var = c(x=1, y=1, z=1))

### vectorized interface
f <- function(x) sum(x^3)
laplacian(f = f, var = c(1, 1, 1))

### symbolic vector-valued functions
f <- array(c("x^2","x*y","x*y","y^2"), dim = c(2,2))
laplacian(f = f, var = c("x","y"))

### numerical vector-valued functions
f <- function(x, y) array(c(x^2,x*y,x*y,y^2), dim = c(2,2))
laplacian(f = f, var = c(x=0,y=0))

### binary operator
"x^3+y^3+z^3" \%laplacian\% c("x","y","z")

}
\references{
Guidotti, E. (2020). "calculus: High dimensional numerical and symbolic calculus in R". \url{https://arxiv.org/abs/2101.00086}
}
\seealso{
Other differential operators: 
\code{\link{curl}()},
\code{\link{derivative}()},
\code{\link{divergence}()},
\code{\link{gradient}()},
\code{\link{hessian}()},
\code{\link{jacobian}()}
}
\concept{differential operators}
