% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{ComputeConfIntervals}
\alias{ComputeConfIntervals}
\title{Pointwise Confidence Intervals under Current Status data}
\usage{
ComputeConfIntervals(data, x, alpha, bw)
}
\arguments{
\item{data}{Dataframe with three variables:
\describe{
    \item{t}{Observation points t sorted in ascending order. All observations need to be positive. The total number of unique observation points equals \code{length(t)}.}
    \item{freq1}{Frequency of observation t satisfying \eqn{x \le t}.
                  The total number of observations with censoring indicator \eqn{\delta =1} equals \code{sum(freq1)}. }
    \item{freq2}{Frequency of observation t. The sample size equals \code{sum(freq2)}. If no tied observations are present in the data \code{length(t)} equals \code{sum(freq2)}. }
}}

\item{x}{numeric vector containing the points where the confidence intervals are computed.
This vector needs to be contained within the observation interval: \eqn{t[1] < min(x) \le max(x) < t[n]}.}

\item{alpha}{confidence level of pointwise confidence intervals.}

\item{bw}{numeric vector of size \code{length(x)}.
This vector contains the pointwise bandwidth values for each point in the vector x.}
}
\value{
List with 5 variables:

\describe{
    \item{MLE }{Maximum Likelihood Estimator. This is a matrix of dimension (m+1)x2 where m is the number of jump points of the MLE.
                 The first column consists of the point zero and the jump locations of the MLE. The second column contains the value zero and the values of the MLE at the jump points. }
    \item{SMLE }{Smoothed Maximum Likelihood Estimator. This is a vector of size \code{length(x)} containing the values of the SMLE for each point in the vector x. }
    \item{CI }{pointwise confidence interval. This is a matrix of dimension \code{length(x)}x2.
               The first resp. second column contains the lower resp. upper values of the confidence intervals for each point in x.}
    \item{Studentized}{points in x for which Studentized nonparametric bootstrap confidence intervals are computed. }
    \item{NonStudentized}{points in x for which classical nonparametric bootstrap confidence intervals are computed.}
}
}
\description{
The function ComputeConfIntervals computes pointwise confidence intervals for the distribution function under current status data.
The confidence intervals are based on the Smoothed Maximum likelihood Estimator and constructed using the nonparametric bootstrap.
}
\details{
In the current status model, the variable of interest \eqn{X} with distribution function \eqn{F} is not observed directly.
A censoring variable \eqn{T} is observed instead together with the indicator \eqn{\Delta = (X \le T)}.
ComputeConfIntervals computes the pointwise \code{1-alpha} bootstrap confidence intervals around the SMLE of \eqn{F} based on a sample of size \code{n <- sum(data$freq2)}.

The bandwidth parameter vector that minimizes the pointwise Mean Squared Error using the subsampling principle in combination with undersmoothing is returned by the function \code{\link{ComputeBW}}.

The default method for constructing the confidence intervals in [Groeneboom & Hendrickx (2017)] is based on estimating the asymptotic variance of the SMLE.
When the bandwidth is small for some point in x, the variance estimate of the SMLE at this point might not exist.
If this happens the Non-Studentized confidence interval is returned for this particular point in x.
}
\examples{
library(Rcpp)
library(curstatCI)

# sample size
n <- 1000

# Uniform data  U(0,2)
set.seed(2)
y <- runif(n,0,2)
t <- runif(n,0,2)
delta <- as.numeric(y <= t)

A<-cbind(t[order(t)], delta[order(t)], rep(1,n))

# x vector
grid<-seq(0.1,1.9 ,by = 0.1)

# data-driven bandwidth vector
bw <- ComputeBW(data =A, x = grid)

# pointwise confidence intervals at grid points:
out<-ComputeConfIntervals(data = A,x =grid,alpha = 0.05, bw = bw)

left <- out$CI[,1]
right <- out$CI[,2]

plot(grid, out$SMLE,type ='l', ylim=c(0,1), main= "",ylab="",xlab="",las=1)
points(grid, left, col = 4)
points(grid, right, col = 4)
segments(grid,left, grid, right)

}
\references{
Groeneboom, P. and Hendrickx, K. (2017). The nonparametric bootstrap for the current status model. Electronic Journal of Statistics 11(2):3446-3848.
}
\seealso{
\code{vignette("curstatCI")}
}
