% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cv-grpsel.R
\name{cv.grpsel}
\alias{cv.grpsel}
\title{Cross-validated group subset selection}
\usage{
cv.grpsel(
  x,
  y,
  group = seq_len(ncol(x)),
  penalty = c("grSubset", "grSubset+grLasso", "grSubset+Ridge"),
  loss = c("square", "logistic"),
  lambda = NULL,
  gamma = NULL,
  nfold = 10,
  folds = NULL,
  cv.loss = NULL,
  cluster = NULL,
  ...
)
}
\arguments{
\item{x}{a predictor matrix}

\item{y}{a response vector}

\item{group}{a vector of length \code{ncol(x)} with the jth element identifying the group that
the jth predictor belongs to; alternatively, a list of vectors with the kth vector identifying
the predictors that belong to the kth group (useful for overlapping groups)}

\item{penalty}{the type of penalty to apply; one of 'grSubset', 'grSubset+grLasso', or
'grSubset+Ridge'}

\item{loss}{the type of loss function to use; 'square' for linear regression or 'logistic' for
logistic regression}

\item{lambda}{an optional list of decreasing sequences of group subset parameters; the list
should contain a vector for each value of \code{gamma}}

\item{gamma}{an optional decreasing sequence of group lasso or ridge parameters}

\item{nfold}{the number of cross-validation folds}

\item{folds}{an optional vector of length \code{nrow(x)} with the ith entry identifying the fold
that the ith observation belongs to}

\item{cv.loss}{an optional cross-validation loss-function to use; should accept a vector of
predicted values and a vector of actual values}

\item{cluster}{an optional cluster for running cross-validation in parallel; must be set up using
\code{parallel::makeCluster}; each fold is evaluated on a different node of the cluster}

\item{...}{any other arguments for \code{grpsel()}}
}
\value{
An object of class \code{cv.grpsel}; a list with the following components:
\item{cv.mean}{a list of vectors containing cross-validation means per value of \code{lambda};
an individual vector in the list for each value of \code{gamma}}
\item{cd.sd}{a list of vectors containing cross-validation standard errors per value of
\code{lambda}; an individual vector in the list for each value of \code{gamma}}
\item{lambda}{a list of vectors containing the values of \code{lambda} used in the fit; an
individual vector in the list for each value of \code{gamma}}
\item{gamma}{a vector containing the values of \code{gamma} used in the fit}
\item{lambda.min}{the value of \code{lambda} minimising \code{cv.mean}}
\item{gamma.min}{the value of \code{gamma} minimising \code{cv.mean}}
\item{fit}{the fit from running \code{grpsel()} on the full data}
}
\description{
Fits the regularisation surface for a regression model with a group subset selection
penalty and then cross-validates this surface.
}
\details{
When \code{loss='logistic'} stratified cross-validation is used to balance
the folds.
}
\examples{
# Grouped data
set.seed(123)
n <- 100
p <- 10
g <- 5
group <- rep(1:g, each = p / g)
beta <- numeric(p)
beta[which(group \%in\% 1:2)] <- 1
x <- matrix(rnorm(n * p), n, p)
y <- x \%*\% beta + rnorm(n)
newx <- matrix(rnorm(p), ncol = p)

# Group subset selection
fit <- cv.grpsel(x, y, group)
plot(fit)
coef(fit)
predict(fit, newx)

# Group subset selection with group lasso shrinkage
fit <- cv.grpsel(x, y, group, penalty = 'grSubset+grLasso')
plot(fit)
coef(fit)
predict(fit, newx)

# Group subset selection with ridge shrinkage
fit <- cv.grpsel(x, y, group, penalty = 'grSubset+Ridge')
plot(fit)
coef(fit)
predict(fit, newx)

# Parallel cross-validation
cl <- parallel::makeCluster(2)
fit <- cv.grpsel(x, y, group, cluster = cl)
parallel::stopCluster(cl)
}
\author{
Ryan Thompson <ryan.thompson@monash.edu>
}
