% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/DroppingInterval.R
\name{estinterval}
\alias{estinterval}
\title{Estimate interval model accounting for missed arrival observations}
\usage{
estinterval(data, mu = median(data), sigma = sd(data)/2, p = 0.2,
  N = 5L, fun = "gamma", trunc = c(0, Inf), fpp = (if (fpp.method ==
  "fixed") 0 else 0.1), fpp.method = "auto", p.method = "auto",
  conf.level = 0.9, group = NA, sigma.within = NA, iter = 10,
  tol = 0.001, silent = F, ...)
}
\arguments{
\item{data}{A numeric list of intervals.}

\item{mu}{Start value for the numeric optimization for the mean arrival interval.}

\item{sigma}{Start value for the numeric optimization for the standard deviation of the arrival interval.}

\item{p}{Start value for the numeric optimization for the probability to not observe an arrival.}

\item{N}{Maximum number of missed observations to be taken into account (default N=5).}

\item{fun}{Assumed distribution for the intervals, one of "\code{normal}" or "\code{gamma}", corresponding
to the \link[stats]{Normal} and \link[stats]{GammaDist} distributions}

\item{trunc}{Use a truncated probability density function with range \code{trunc}}

\item{fpp}{Baseline proportion of intervals distributed as a random poisson process with mean arrival interval \code{mu}}

\item{fpp.method}{A string equal to 'fixed' or 'auto'. When 'auto' \code{fpp} is optimized as a free model parameter,
in which case \code{fpp} is taken as start value in the optimisation}

\item{p.method}{A string equal to 'fixed' or 'auto'. When 'auto' \code{p} is optimized as a free model parameter,
in which case \code{p} is taken as start value in the optimisation}

\item{conf.level}{Confidence level for deviance test that checks whether model with nonzero missed event probability
\code{p} significantly outperforms a model without a missed event probability (\code{p=0}).}

\item{group}{optional vector of equal length as data, indicating the group or subject in which the interval was observed}

\item{sigma.within}{optional within-subject standard deviation. When equal to default 'NA', assumes
no additional between-subject effect, with \code{sigma.within} equal to \code{sigma}. When equal to 'auto'
an estimate is provided by iteratively calling \link[intRvals]{partition}}

\item{iter}{maximum number of iterations in numerical iteration for \code{sigma.within}}

\item{tol}{tolerance in the iteration, when \code{sigma.within} changes less than this value in one iteration step, the optimization is considered converged.}

\item{silent}{logical. When \code{TRUE} print no information to console}

\item{...}{Additional arguments to be passed to \link[stats]{optim}}
}
\value{
This function returns an object of class \code{intRvals}, which is a list containing the following:
\describe{
  \item{\code{data}}{the interval data}
  \item{\code{mu}}{the modelled mean interval}
  \item{\code{mu.se}}{the modelled mean interval standard error}
  \item{\code{sigma}}{the modelled interval standard deviation}
  \item{\code{p}}{the modelled probability to not observe an arrival}
  \item{\code{fpp}}{the modelled fraction of arrivals following a random poisson process, see \link[intRvals]{intervalpdf}}
  \item{\code{N}}{the highest number of consecutive missed arrivals taken into account, see \link[intRvals]{intervalpdf}}
  \item{\code{convergence}}{convergence field of \link[stats]{optim}}
  \item{\code{counts}}{counts field of \link[stats]{optim}}
  \item{\code{loglik}}{vector of length 2, with first element the log-likelihood of the fitted model, and second element the log-likelihood of the model without a missed event probability (i.e. \code{p}=0)}
  \item{\code{df.residual}}{degrees of freedom, a 2-vector (1, number of intervals - \code{n.param})}
  \item{\code{n.param}}{number of optimized model parameters}
  \item{\code{p.chisq}}{p value for a likelihood-ratio test of a model including a miss probability relative against a model without a miss probability}
  \item{\code{distribution}}{assumed interval distribution, one of 'gamma' or 'normal'}
  \item{\code{trunc}}{interval range over which the interval pdf was truncated and normalized}
  \item{\code{fpp.method}}{A string equal to 'fixed' or 'auto'. When 'auto' \code{fpp} has been optimized as a free model parameter}
  \item{\code{p.method}}{A string equal to 'fixed' or 'auto'. When 'auto' \code{p} has been optimized as a free model parameter}
}
}
\description{
Estimate interval mean and variance accounting for missed arrival observations,
by fitting the probability density function \link[intRvals]{intervalpdf} to the interval data.
}
\details{
The probability density function for observed intervals \link[intRvals]{intervalpdf}
is fit to \code{data} by maximization of the
associated log-likelihood using \link[stats]{optim}.

Within-group variation \code{sigma.within} may be separated from the total variation \code{sigma} in an iterative fit of \link[intRvals]{intervalpdf} on the interval data.
In the iteration \link[intRvals]{partition} is used to (1) determine which intervals according to the fit are a fundamental interval at a confidence level \code{conf.level},
and (2) to partition the within-group variation from the total variation in interval length.

Within- and between-group variation is estimated on the subset of fundamental intervals with repeated measures only.
As the set of fundamental interval depends on the precise value of \code{sigma.within}, the fit of \link[intRvals]{intervalpdf} and the subsequent estimation of
\code{sigma.within} using \link[intRvals]{partition} is iterated until both converge to a stable solution. Parameters \code{tol}
and \code{iter} set the threshold for convergence and the maximum number of iterations.

We note that an exponential interval model can be fitted by setting \code{fpp=1} and \code{fpp.method=fixed}.
}
\examples{
data(goosedrop)
# calculate mean and standard deviation of arrival intervals, accounting for missed observations:
dr=estinterval(goosedrop$interval)
# plot some summary information
summary(dr)
# plot a histogram of the intervals and fit:
plot(dr)
# test whether the mean arrival interval is greater than 200 seconds:
ttest(dr,mu=200,alternative="greater")

# let's estimate mean and variance of dropping intervals by site
# (schiermonnikoog vs terschelling) for time period 5.
# first prepare the two datasets:
set1=goosedrop[goosedrop$site=="schiermonnikoog" & goosedrop$period == 5,]
set2=goosedrop[goosedrop$site=="terschelling"  & goosedrop$period == 5,]
# allowing a fraction of intervals to be distributed randomly (fpp='auto')
dr1=estinterval(set1$interval,fpp.method='auto')
dr2=estinterval(set2$interval,fpp.method='auto')
# plot the fits:
plot(dr1,xlim=c(0,1000))
plot(dr2,xlim=c(0,1000))
# mean dropping interval are not significantly different
# at the two sites (on a 0.95 confidence level):
ttest(dr1,dr2)
# now compare this test with a t-test not accounting for unobserved intervals:
t.test(set1$interval,set2$interval)
# not accounting for missed observations leads to a (spurious)
# larger difference in means, which also increases
# the apparent statistical significance of the difference between means
}
