# This function computes a matrix of log-likelihoods for obtaining the observed item responses across examinees given theta values
# This function is used for MMLE-EM algorithm
likelihood <- function(meta, data1_drm=NULL, data2_drm=NULL, data_plm=NULL, theta, D=1) {

  # compute the log-likelihoods given the quadrature points
  # when there are dichotomous items
  if(!is.null(data1_drm)) {

    # compute the probabilities of answering correctly on items
    ps <- drm(theta=theta, a=meta$drm$a, b=meta$drm$b, g=meta$drm$g, D=D)
    if(length(theta) == 1L) ps <- matrix(ps)

    # compute the probabilities of answering incorrectly on items
    qs <- 1 - ps

    # to prevent that log(ps) and log(qs) have -Inf values
    log_ps <- suppressWarnings(log(ps))
    log_qs <- suppressWarnings(log(qs))
    log_ps <- ifelse(is.nan(log_ps), log(1e-20), log_ps)
    log_qs <- ifelse(is.nan(log_qs), log(1e-20), log_qs)
    log_ps <- ifelse(is.infinite(log_ps), log(1e-20), log_ps)
    log_qs <- ifelse(is.infinite(log_qs), log(1e-20), log_qs)

    # compute the loglikelihood values for all examinees at each quadrature point
    # a row indicate the examinee and column indicate the quad point
    llike_drm <- tcrossprod(x=data1_drm, y=log_ps) + tcrossprod(x=data2_drm, y=log_qs)
  } else {
    llike_drm <- 0L
  }

  # when there are polytomous items
  if(!is.null(data_plm)) {

    # extract polytomous model info
    # model <- meta$plm$model

    # make a list of arguments
    # args <- list(meta$plm$a, meta$plm$d, model)

    # compute the category probabilities of items
    # prob.plm <-
    #   purrr::pmap(.l=args, .f=plm, theta=theta, D=D) %>%
    #   do.call(what='cbind')
    # compute the category probabilities of items
    prob.plm <- vector('list', length(meta$plm$a))
    for(k in 1:length(meta$plm$a)) {
      prob.plm[[k]] <- plm(theta=theta, a=meta$plm$a[k], d=meta$plm$d[[k]], D=D, pmodel=meta$plm$model[k])
      if(length(theta) == 1L) prob.plm[[k]] <- rbind(prob.plm[[k]])
    }
    prob.plm <- do.call(prob.plm, what='cbind')

    # to prevent that log(prob.plm) have NaN or -Inf values
    log_prob.plm <- suppressWarnings(log(prob.plm))
    log_prob.plm <- ifelse(is.nan(log_prob.plm), log(1e-20), log_prob.plm)
    log_prob.plm <- ifelse(is.infinite(log_prob.plm), log(1e-20), log_prob.plm)

    # compute the loglikelihood values for all examinees at each quadrature point
    # a row indicate the examinee and column indicate the quad point
    llike_plm <- tcrossprod(x=data_plm, y=log_prob.plm)
  } else {
    llike_plm <- 0L
  }

  # sum of log-likelihood matrix
  LL <- llike_drm + llike_plm

  # transform to likelihood matrix
  L <- exp(LL)

  # return results
  rst <- list(L=L, LL=LL)
  rst


}

