% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_irtfit.R
\name{plot.irtfit}
\alias{plot.irtfit}
\title{Draw raw and standardized residual plots}
\usage{
\method{plot}{irtfit}(
  x,
  item.loc = NULL,
  type = "both",
  ci.method = c("wald", "cp", "wilson", "wilson.cr"),
  show.table = TRUE,
  layout.col = 2,
  xlab.text,
  ylab.text,
  main.text,
  lab.size = 15,
  main.size = 15,
  axis.size = 15,
  line.size = 1,
  point.size = 2.5,
  strip.size = 12,
  ylim.icc = c(0, 1),
  ylim.sr.adjust = FALSE,
  ylim.sr = c(-4, 4),
  ...
)
}
\arguments{
\item{x}{An object of class \code{\link{irtfit}}.}

\item{item.loc}{An integer value indicating that the \emph{n}th item (or the location of the item) is plotted. See below for
details.}

\item{type}{A character string indicating what type of residual plot is returned. Available options
are "icc" for the raw residual plot, "sr" for the standardized residual plot, and "both" for both of them.
Default is "both".}

\item{ci.method}{A character string indicating what method is used to estimate the confidence interval for the raw residual plot.
Available options are "wald" for Wald method, "cp" for Clopper-Pearson interval, "wilson" for Wilson score interval, and
"wilson.cr" for Wilson score interval with continuity correction. Default is "wald". See below for details.}

\item{show.table}{A logical value. If TRUE, a contingency table containing the information used to draw the residual
plots for the studied item is returned. This contingency table is the same as one contained in the internal object of \code{contingency.plot}
in the object of class \code{\link{irtfit}}. Default is TRUE.}

\item{layout.col}{An integer value indicating the number of columns in the panel when a polytomous item is used.
Default is 2.}

\item{xlab.text}{A title for the x axis. If missing, the default string is used.}

\item{ylab.text}{A title for the y axis. If \code{type = "both"}, two character strings can be
specified for the raw residual and standardized residual plots, respectively. If missing,
the default strings are used.}

\item{main.text}{An overall title for the plot. If \code{type = "both"}, two character strings
can be specified for the raw residual and standardized residual plots, respectively. If missing,
the default strings are used.}

\item{lab.size}{The size of xlab and ylab. Default is 15.}

\item{main.size}{The size of \code{main.text}. Default is 15.}

\item{axis.size}{The size of labels along the x and y axes. Default is 15.}

\item{line.size}{The size of lines. Default is 1.}

\item{point.size}{The size of points. Default is 2.5.}

\item{strip.size}{The size of facet labels. Default is 12.}

\item{ylim.icc}{A vector of two numeric values specifying the range of y axis for the raw residual plot. Default is c(0, 1).}

\item{ylim.sr.adjust}{A logical value. If TRUE, the range of y axis for the standardized residual plot is adjusted for each item.
If FALSE, the range of y axis for the standardized residual plot is fixed to the values specified in the argument \code{ylim.sr}.}

\item{ylim.sr}{A vector of two numeric values specifying the range of y axis for the standardized residual plot.
Default is c(-4, 4).}

\item{...}{Further arguments passed from the function \code{ggplot()} in the \pkg{ggplot2} package.}
}
\description{
This function provides graphical displays to look at residuals between the observed data
and model-based predictions (Hambleton, Swaminathan, & Rogers, 1991). This function gives two residual plots for
each score category of an item: (a) the raw residual plot and (b) the standardized residual plot. Note that
for dichotomous items the residual plots are drawn only for the score category of 1.
}
\details{
All of the plots are drawn using the ggplot2 package.

Once the results of the IRT model fit analysis are obtained from the function \code{\link{irtfit}},
an object of class \code{\link{irtfit}} can be used to draw the IRT raw residual and standardized residual plots. Especially, the information
contained in an internal object of \code{contingency.plot} are mainly used to draw the residual plots.

Because the residual plots are drawn for an item at a time, you have to indicate which item will be evaluated. For this,
you should specify an integer value, which is the location of the studied item, in the argument \code{item.loc}.
For example, if you want to draw the residual plots for the third item, then \code{item.loc = 3}.

In terms of the raw residual plot, the argument \code{ci.method} is used to select a method to estimate the confidence intervals
among four methods. Those methods are "wald" for the Wald interval, which is based on the normal approximation (Laplace, 1812),
"cp" for Clopper-Pearson interval (Clopper & Pearson, 1934), "wilson" for Wilson score interval (Wilson, 1927), and
"wilson.cr" for Wilson score interval with continuity correction (Newcombe, 1998).
See \url{https://en.wikipedia.org/wiki/Binomial_proportion_confidence_interval} for more details about
the binomial proportion confidence intervals. Note that the width of confidence interval is determined by the \eqn{\alpha}-level
specified in the argument \code{alpha} of the function \code{\link{irtfit}}.

Regarding the standardized residual plot, any standardized residuals greater than the specified criterion value
in the argument {\code{overSR}} of the function \code{\link{irtfit}} are displayed as triangles. Otherwise,
they are displayed as circles.
}
\examples{
## import the "-prm.txt" output file from flexMIRT
flex_sam <- system.file("extdata", "flexmirt_sample-prm.txt", package = "irtplay")

# select the first two dichotomous items and last polytomous item
x <- bring.flexmirt(file=flex_sam, "par")$Group1$full_df[c(1:2, 55), ]

# generate examinees' abilities from N(0, 1)
set.seed(23)
score <- rnorm(1000, mean=0, sd=1)

# simulate the response data
data <- simdat(x=x, theta=score, D=1)

\donttest{
# compute fit statistics
fit <- irtfit(x=x, score=score, data=data, group.method="equal.freq",
               n.width=11, loc.theta="average", range.score=c(-4, 4), D=1, alpha=0.05, overSR=1.5)

# residual plots for the first item (dichotomous item)
plot(x=fit, item.loc=1, type = "both", ci.method = "wald", show.table=TRUE, ylim.sr.adjust=TRUE)

# residual plots for the third item (polytomous item)
plot(x=fit, item.loc=3, type = "both", ci.method = "wald", show.table=FALSE, ylim.sr.adjust=TRUE)

# raw residual plot for the third item (polytomous item)
plot(x=fit, item.loc=3, type = "icc", ci.method = "wald", show.table=TRUE, ylim.sr.adjust=TRUE)

# standardized residual plot for the third item (polytomous item)
plot(x=fit, item.loc=3, type = "sr", ci.method = "wald", show.table=TRUE, ylim.sr.adjust=TRUE)
}

}
\references{
Clopper, C. J., & Pearson, E. S. (1934). The use of confidence or fiducial limits illustrated in the case of the binomial.
\emph{Biometrika, 26}(4), 404-413.

Hambleton, R. K., Swaminathan, H., & Rogers, H. J. (1991).\emph{Fundamentals of item response theory}. Newbury Park, CA: Sage.

Laplace, P. S. (1820).\emph{Theorie analytique des probabilites} (in French). Courcier.

Newcombe, R. G. (1998). Two-sided confidence intervals for the single proportion: comparison of seven methods.
\emph{Statistics in medicine, 17}(8), 857-872.

Wilson, E. B. (1927). Probable inference, the law of succession, and statistical inference.
\emph{Journal of the American Statistical Association, 22}(158), 209-212.
}
\seealso{
\code{\link{irtfit}}
}
\author{
Hwanggyu Lim \email{hglim83@gmail.com}
}
