% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/assign_date.R
\name{assign_date}
\alias{assign_date}
\title{Assign dates to two sequential hours}
\usage{
assign_date(start, end, ambiguity = 0)
}
\arguments{
\item{start, end}{A \code{hms} or \code{POSIXt} object indicating the start or end
hour.}

\item{ambiguity}{(optional) a \code{numeric} or \code{NA} value to instruct
\code{assign_date()} on how to deal with ambiguities (see Details) (default:
\code{0}).}
}
\value{
A \code{start}--\code{end} \code{Interval} object.
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#maturing}{\figure{lifecycle-maturing.svg}{options: alt='[Maturing]'}}}{\strong{[Maturing]}}

\code{assign_date()} assign dates to two sequential hours. It can facilitate
time arithmetic by locating time values without a date reference on a
timeline.
}
\details{
\subsection{Class requirements}{

The \code{mctq} package works with a set of object classes specially created to
hold time values. These classes can be found in the
\link[lubridate:lubridate-package]{lubridate} and \link[hms:hms-package]{hms}
packages. Please refer to those package documentations to learn more about
them.
}

\subsection{\code{ambiguity} argument}{

In cases when \code{start} is equal to \code{end}, there are two possibilities of
intervals between the two hours (ambiguity). That's because \code{start} and \code{end}
can be at the same point in time or they can distance themselves by one day,
considering a two-day timeline.\preformatted{ start,end       start,end       start,end       start,end
   start            end            start            end
   10:00           10:00           10:00           10:00
-----|---------------|---------------|---------------|----->
    0h              0h              0h              0h
            24h             24h             24h
}

You must instruct \code{assign_date()} on how to deal with this problem if it
occurs. There are three options to choose.
\itemize{
\item \code{ambiguity = 0}: to consider the interval between \code{start} and \code{end} as 0
hours, i.e., \code{start} and \code{end} are located at the same point in time
(default).
\item \code{ambiguity = 24}: to consider the interval between \code{start} and \code{end} as 24
hours, i.e., \code{start} and \code{end} distance themselves by one day.
\item \code{ambiguity = NA}: to disregard these cases, assigning \code{NA} as value.
}
}

\subsection{Base date and timezone}{

\code{assign_date()} uses the
\href{https://en.wikipedia.org/wiki/Unix_time}{Unix epoch} (1970-01-01) date as
the start date for creating intervals.

The output will always have \code{"UTC"} set as timezone. Learn more about
time zones in \link[base:timezones]{base::timezone}.
}

\subsection{\code{POSIXt} objects}{

\code{\link[base:as.POSIXlt]{POSIXt}} objects passed as argument to \code{start} or \code{end}
will be stripped of their dates. Only the time will be considered.

Both \code{POSIXct} and \code{POSIXlt} are objects that inherits the class \code{POSIXt}.
Learn more about it in \link[base:DateTimeClasses]{base::DateTimeClasses}.
}

\subsection{\code{NA} values}{

\code{assign_date()} will return an \code{Interval} \code{NA}-\code{NA} if \code{start} or \code{end} are
\code{NA}.
}
}
\examples{
## Scalar example

start <- hms::parse_hms("23:11:00")
end <- hms::parse_hms("05:30:00")
assign_date(start, end)
#> [1] 1970-01-01 23:11:00 UTC--1970-01-02 05:30:00 UTC # Expected

start <- hms::parse_hms("10:15:00")
end <- hms::parse_hms("13:25:00")
assign_date(start, end)
#> [1] 1970-01-01 10:15:00 UTC--1970-01-01 13:25:00 UTC # Expected

start <- hms::parse_hms("05:42:00")
end <- hms::as_hms(NA)
assign_date(start, end)
#> [1] NA--NA # Expected

## Vector example

start <- c(hms::parse_hm("09:45"), hms::parse_hm("20:30"))
end <- c(hms::parse_hm("21:15"), hms::parse_hm("04:30"))
assign_date(start, end)
#> [1] 1970-01-01 09:45:00 UTC--1970-01-01 21:15:00 UTC # Expected
#> [2] 1970-01-01 20:30:00 UTC--1970-01-02 04:30:00 UTC # Expected

## To assign a 24 hours interval to ambiguities

start <- lubridate::as_datetime("1985-01-15 12:00:00")
end <- lubridate::as_datetime("2020-09-10 12:00:00")
assign_date(start, end, ambiguity = 24)
#> [1] 1970-01-01 12:00:00 UTC--1970-01-02 12:00:00 UTC # Expected
}
\seealso{
Other utility functions: 
\code{\link{cycle_time}()},
\code{\link{pretty_mctq}()},
\code{\link{qplot_walk}()},
\code{\link{random_mctq}()},
\code{\link{raw_data}()},
\code{\link{round_time}()},
\code{\link{shorter_interval}()},
\code{\link{sum_time}()}
}
\concept{utility functions}
