% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/IHS.Algorithm.R
\name{HS}
\alias{HS}
\title{Optimization using Harmony Search Algorithm}
\usage{
HS(FUN, optimType = "MIN", numVar, numPopulation = 40, maxIter = 500,
  rangeVar, PAR = 0.3, HMCR = 0.95, bandwith = 0.05)
}
\arguments{
\item{FUN}{an objective function or cost function,}

\item{optimType}{a string value that represent the type of optimization.
There are two option for this arguments: \code{"MIN"} and \code{"MAX"}.
The default value is \code{"MIN"}, which the function will do minimization.
Otherwise, you can use \code{"MAX"} for maximization problem.
The default value is \code{"MIN"}.}

\item{numVar}{a positive integer to determine the number variables.}

\item{numPopulation}{a positive integer to determine the number populations. The default value is 40.}

\item{maxIter}{a positive integer to determine the maximum number of iterations. The default value is 500.}

\item{rangeVar}{a matrix (\eqn{2 \times n}) containing the range of variables,
where \eqn{n} is the number of variables, and first and second rows
are the lower bound (minimum) and upper bound (maximum) values, respectively.
If all variable have equal upper bound, you can define \code{rangeVar} as
matrix (\eqn{2 \times 1}).}

\item{PAR}{a positive integer to determine the value of Pinch Adjusting Ratio. The default value is 0.3.}

\item{HMCR}{a positive integer to determine the Harmony Memory Consideration Rate. The default value is 0.95.}

\item{bandwith}{a positive integer to determine the bandwith. The default value is 0.05.}
}
\value{
\code{Vector [v1, v2, ..., vn]} where \code{n} is number variable
        and \code{vn} is value of \code{n-th} variable.
}
\description{
This is the internal function that implements Improved Harmony Search
Algorithm. It is used to solve continuous optimization tasks.
Users do not need to call it directly,
but just use \code{\link{metaOpt}}.
}
\details{
Harmony Search (HS)  was proposed by (Geem et al., 2001)
mimicking the improvisation of music players. Furthermore,
Improved Harmny Search (HS), proposed by Mahdavi, employs a method for
generating new solution vectors that enhances accuracy and convergence
rate of harmony search algorithm.

In order to find the optimal solution, the algorithm follow the following steps.
\itemize{
\item Step 1. Initialized the problem and algorithm parameters
\item Step 2. Initialize the Harmony Memory, creating the Harmony memory and give
      random rumber for each memory.
\item Step 3. Improvise new Harmony, Generating new Harmony based on parameter defined by user
\item Step 4. Update the Harmony Memory, If new harmony have better fitness than the worst harmony in
      Harmony Memory, then replace the worst harmony with new Harmony.
\item Step 5. Check termination criteria, if termination criterion is satisfied, return the
      best Harmony as the optimal solution for given problem. Otherwise, back to Step 3.
}
}
\examples{
##################################
## Optimizing the quartic with noise function

# define Quartic with noise function as objective function
quartic <- function(x){
    dim <- length(x)
    result <- sum(c(1:dim)*(x^4))+runif(1)
    return(result)
}

## Define parameter
numVar <- 5
rangeVar <- matrix(c(-10,10), nrow=2)
PAR <- 0.3
HMCR <- 0.95
bandwith <- 0.05

## calculate the optimum solution using Harmony Search algorithm
resultHS <- HS(quartic, optimType="MIN", numVar, numPopulation=20,
                 maxIter=100, rangeVar, PAR, HMCR, bandwith)

## calculate the optimum value using quartic with noise function
optimum.value <- quartic(resultHS)

}
\references{
Geem, Zong Woo, Joong Hoon Kim, and G. V. Loganathan (2001). "A new
heuristic optimization algorithm: harmony search." Simulation 76.2: pp. 60-68.

M. Mahdavi, M. Fesanghary, E. Damangir, An improved harmony search algorithm
for solving optimization problems, Applied Mathematics and Computation,
Volume 188, Issue 2, 2007, Pages 1567-1579, ISSN 0096-3003,
https://doi.org/10.1016/j.amc.2006.11.033
}
\seealso{
\code{\link{metaOpt}}
}
