% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/optint.R
\name{optint}
\alias{optint}
\title{Optimal intervention}
\usage{
optint(Y, X, control = NULL, wgt = rep(1, length(Y)),
  method = "non-parametric", lambda = 100, sigma = 1,
  grp.size = 30, n.boot = 1000, sign.factor = 2/3, alpha = 0.05,
  n.quant = floor(length(Y)/10), perm.test = T, n.perm = 1000,
  quick = F, plot = T, seed = runif(1, 0, .Machine$integer.max))
}
\arguments{
\item{Y}{outcome vector (must be numeric without NA's).}

\item{X}{numeric data frame or matrix of factors to be considered.}

\item{control}{numeric data frame or matrix of factors to control for. these are factors
that we can't consider while looking for the optimal intervention
(e.g. race).}

\item{wgt}{an optional vector of weights.}

\item{method}{the method to be used. either "non-parametric" (default), "correlation" or
"nearest-neighbors".}

\item{lambda}{the lagrange multiplier. also known as the shadow price of an
intervention.}

\item{sigma}{distance penalty for the nearest-neighbors method.}

\item{grp.size}{for the nearest-neighbors method; if the number of examples in each
control group is smaller than grp.size, performs weight adjustment
using \code{\link[optinterv]{wgt_adjust}}. else,
calculate weights seperatly for each control group.}

\item{n.boot}{number of bootstrap replications to use for the standard errors /
confidence intervals calculation.}

\item{sign.factor}{what proportion of quantiles should to be increased (decreased)
in order to return a positive (negative) sign? not relevant for
the correlation method (there the correlation sign is returned).}

\item{alpha}{significance level for the confidence intervals.}

\item{n.quant}{number of quantiles to use when calculating CDF distance.}

\item{perm.test}{logical. if TRUE (default) performs permutation test and calculates p-values.}

\item{n.perm}{number of permutations for the permutation test.}

\item{quick}{logical. if TRUE, returns only \eqn{E(X | I=1) - E(X | I=0)} as an estimate.
this estimate is used by \code{\link{optint_by_group}}.}

\item{plot}{logical. if TRUE (default), the results are plotted by \code{\link{plot.optint}}.}

\item{seed}{the seed of the random number generator.}
}
\value{
an object of class "optint". This object is a list containing
        the folowing components:
 \item{estimates}{standardized point estimates (correlations for the
 correlation method and cdf distances otherwise).}
 \item{estimates_sd}{estimates standard deviation.}
 \item{details}{a list containing further details, such as:}
 \itemize{
  \item Y_diff - \eqn{E(Y | I=1) - E(Y | I=0)}.
  \item Y_diff_sd - standard deviation for Y_diff.
  \item method - the method used.
  \item lambda - the lagrange multiplier used.
  \item signs - signs (i.e. directions) for the estimates.
  \item p_value - p-values for the estimates.
  \item ci - a matrix of confidence intervals for the estimates.
  \item stand_factor - the standardization factor used to standardize the results.
  \item kl_distance - the Kullback–Leibler divergence of \eqn{P(X | I=0) from P(X | I=1)}.
  \item new_sample - a data frame containing X, control (if provided),
        wgt (the original weights) and wgt1 (the new weights under \eqn{I = 1}.)
 }

In addition, the function \code{\link[base]{summary}} can be used to
print a summary of the results.
}
\description{
identifies the factors with the greatest potential to increase
a pre-specified outcome, using varius methods.
}
\examples{
# generate data
n <- 50
p <- 3
features <- matrix(rnorm(n*p), ncol = p)
men <- matrix(rbinom(n, 1, 0.5), nrow = n)
outcome <- 2*(features[,1] > 1) + men*pmax(features[,2], 0) + rnorm(n)
outcome <- as.vector(outcome)

#find the optimal intervention using the non-parametric method:
imp_feat <- optint(Y = outcome, X = features, control = men,
                   method = "non-parametric", lambda = 10, plot = TRUE,
                   n.boot = 100, n.perm = 100)

#by default, only the significant features are displayed
#(see ?plot.optint for further details).
#for customized variable importance plot, use plot():
plot(imp_feat, plot.vars = 3)

#show summary of the results using summary():
summary(imp_feat)
}
