#' @title Extract potential benefit of features
#'
#' @description Provides the maximum values of benefits to achieve
#' for each feature given a set of data inputs.
#'
#' @param x `data-class` object.
#'
#' @details For a given feature \eqn{s}, let \eqn{I_s} be the set of planning units associated with \eqn{s},
#' let \eqn{r_{is}} is the amount of feature \eqn{s} in planning unit \eqn{i}, let \eqn{K_{s}} be the
#' set of threats associated with \eqn{s}, and let \eqn{K_{i}} be the set of threats associated with \eqn{i}.
#' The local benefit associated with \eqn{s} in a unit \eqn{i} is given by:
#'
#' \deqn{
#' b_{is} = p_{is} r_{is} \\
#' b_{is} = \frac{ \sum_{k \in K_i \cap K_s}{x_{ik}}}{|K_i \cap K_s|} r_{is}
#' }
#'
#' Where \eqn{x_{ik}} is a decision variable such that \eqn{x_{ik} = 1} if an
#' action againts threat \eqn{k} is applied in unit \eqn{i}, and \eqn{x_{ik} = 0}, otherwise.
#' This expression for the probability of persistence of the feature (\eqn{p_{is}})
#' is defined only for the cases where we work with values of binary intensities
#' (presence or absence of threats). See the [sensitivities](https://prioriactions.github.io/prioriactions/articles/sensitivities.html)
#' vignette to know the work with continuous intensities.
#'
#' While the total benefit is calculated as the sum of the local benefits per feature:
#'
#' \deqn{
#' b_{s} = \sum_{i \in I_{s}}\frac{ \sum_{k \in K_i \cap K_s}{x_{ik}}}{|K_i \cap K_s|} r_{is}
#' }
#'
#' Since the potential benefit is being calculated, all variables \eqn{x_{ik}} are assumed to be equal
#' to 1; that is, all possible actions are carried out, and only those that have a **lock-out**
#' status are kept out of the planning (see `inputData()` function for more information).
#'
#' @return [data.frame].
#'
#' @examples
#' # set seed for reproducibility
#' set.seed(14)
#'
#' ## Load data
#' data(sim_pu_data, sim_features_data, sim_dist_features_data,
#' sim_threats_data, sim_dist_threats_data, sim_sensitivity_data,
#' sim_boundary_data)
#'
#' ## Create data instance
#' problem_data <- inputData(
#'   pu = sim_pu_data, features = sim_features_data, dist_features = sim_dist_features_data,
#'   threats = sim_threats_data, dist_threats = sim_dist_threats_data,
#'   sensitivity = sim_sensitivity_data, boundary = sim_boundary_data
#' )
#'
#' ## Get maximum benefits to obtain
#' getPotentialBenefit(problem_data)
#'
#' @name getPotentialBenefit
NULL

#' @rdname getPotentialBenefit
#' @export
getPotentialBenefit <- function(x) {
  # assert argument is valid
  assertthat::assert_that(inherits(x, "Data"))

  benefits <- rcpp_stats_benefit(x$data$pu,
                                 x$data$features,
                                 x$data$dist_features,
                                 x$data$threats,
                                 x$data$dist_threats,
                                 x$data$sensitivity,
                                 0)
  return(benefits)
}
