% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plainMethods_stoiBuild_stoiCheck.r
\name{stoiCreate}
\alias{stoiCreate}
\title{Stoichiometry Matrix from Reaction Equations}
\usage{
stoiCreate(
  reactions,
  eval = FALSE,
  env = globalenv(),
  toRight = "_forward",
  toLeft = "_backward"
)
}
\arguments{
\item{reactions}{A named vector of character strings, each representing
a (chemical) reaction. See syntax details below.}

\item{eval}{Logical. If \code{FALSE} (default), the returned matrix is of
type \code{\link[base]{character}} and any mathematical expressions are
returned as text. If \code{TRUE}, an attempt is made to return a
\code{\link[base]{numeric}} matrix by evaluating the expression making use
\code{env}.}

\item{env}{Only relevant if \code{eval} is \code{TRUE}. Must be an
environment or list supplying constants, functions, and operators needed to
evaluate expressions in the generated matrix.}

\item{toRight}{Only relevant for reversible reactions. The passed character
string is appended to the name of the respective element of
\code{reactions} to create a unique name for the forward reaction.}

\item{toLeft}{Like \code{toRight}, but this is the suffix for the backward
reaction.}
}
\value{
A matrix with the following properties:
\itemize{
  \item{The number of columns equals the total number of components present
    in \code{reactions}. The components' names are used as column names.}
  \item{The number of rows equals the length of \code{reactions} plus the
    number of reversible reactions. Thus, a single row is created for each
    non-reversible reaction but two rows are created for reversible ones.
    The latter represent the forward and backward reaction (in that order).
    The row names are constructed from the names of \code{reactions}, making
    use of the suffixes \code{toRight} and \code{toLeft} in the case of
    reversible reactions.}
  \item{The matrix is filled with the stoichiometric factors extracted from
    \code{reactions}. Empty elements are set to zero.}
  \item{The type of the matrix (\code{\link[base]{character}} or
    \code{\link[base]{numeric}}) depends on the value of \code{eval}.}
}
}
\description{
Creates a stoichiometry matrix from a set of reaction equations.
}
\note{
The syntax rules for reaction equations are as follows (see examples):
\itemize{
  \item{There must be a left hand side and a right hand side. Sides must be
    separated by one of the arrows '->', '<-', or '<->' with the latter
    indicating a reversible reaction.}
  \item{Names of component(s) must appear at each side of the reaction. These
    must be legal row/column names in R. If multiple
    components are consumed or produced, they must be separated by '+'.}
  \item{Any stoichiometric factors need to appear before the respective
    component name using '*' as the separating character. Stoichiometric
    factors being equal to unity can be omitted.}
  \item{A stoichiometric factor is treated as a mathematical expression.
    In common cases, it is just a numeric constant. However, the
    expression can also involve references to variables or functions. If such
    an expression contains math operators '*' or '+' it needs to be enclosed
    in parenthesis.}
}
}
\examples{
# EXAMPLE 1: From https://en.wikipedia.org/wiki/Petersen_matrix (July 2016)
#
reactions <- c(
  formS=  "A + 2 * B -> S",
  equiES= "E + S <-> ES",
  decoES= "ES -> E + P"
)
stoi <- stoiCreate(reactions, eval=TRUE, toRight="_f", toLeft="_b")
print(stoi)

# EXAMPLE 2: Decomposition of organic matter (selected equations only)
#
# Eq. 1 and 2 are from Soetaert et al. (1996), Geochimica et Cosmochimica
# Acta, 60 (6), 1019-1040. 'OM' is organic matter. Constants 'nc' and 'pc'
# represent the nitrogen/carbon and phosphorus/carbon ratio, respectively.
reactions <- c(
  oxicDegrad= "OM + O2 -> CO2 + nc * NH3 + pc * H3PO4 + H2O",
  denitrific= "OM + 0.8*HNO3 -> CO2 + nc*NH3 + 0.4*N2 + pc*H3PO4 + 1.4*H2O",
  dissPhosp1= "H3PO4 <-> H + H2PO4",
  dissPhosp2= "H2PO4 <-> H + HPO4"
)
# Non-evaluated matrix
stoi <- stoiCreate(reactions, toRight="_f", toLeft="_b")
print(stoi)
# Evaluated matrix ('nc' and 'pc' according to Redfield ratio)
pars <- list(nc=16/106, pc=1/106)
stoi <- stoiCreate(reactions, eval=TRUE, env=pars, toRight="_f", toLeft="_b")
print(stoi)
}
\seealso{
Use \code{\link[rodeo]{stoiCheck}} to validate the mass balance of
  the generated matrix.
}
\author{
David Kneis \email{david.kneis@tu-dresden.de}
}
