#' Set access scopes for OAuth access
#'
#' Define access scopes for an application.
#' See \url{https://developer.typeform.com/get-started/scopes/}
#' for more information on what scopes are allowed and their permissions.
#'
#' @importFrom assertthat assert_that
#' @param scopes A character vector of scopes
#' @export
#' @examples
#' rtypeform_set_scope(scopes = c("forms:read"))
rtypeform_set_scope = function(scopes = NULL) {
  if (!is.null(scopes)) {
    assert_that(is.character(scopes))
    options("rtypeform.scopes_selected" = scopes)
  }
  return(invisible(NULL))
}

#' @description Set option for rtypeform client id, secret
#' @param id A client id from a typeform registered application
#' @rdname rtypeform_set_scope
#' @export
rtypeform_set_client_id = function(id = NULL) {
  if (!is.null(id)) {
    assert_that(is.character(id))
    options("rtypeform.client_id" = id)
  }
  return(invisible(NULL))
}

#' @rdname rtypeform_set_scope
#' @param secret A client secret from a typeform registered application
#' @export
rtypeform_set_client_secret = function(secret = NULL) {
  if (!is.null(secret)) {
    assert_that(is.character(secret))
    options("rtypeform.client_secret" = secret)
  }
  return(invisible(NULL))
}

#' @rdname rtypeform_set_scope
#' @export
#' @param token Either a string of a personal access token,
#' or an OAuth token returned by \code{make_new_token}.
rtypeform_set_token = function(token) {
  # if the token is an ouath token generated by make_new_token
  if (inherits(token, "Token2.0")) {
    Sys.setenv("typeform_api2" = token$credentials$access_token)
  } else if (is.character(token)) {
    # if it's a personal token
    Sys.setenv("typeform_api2" = token)
  } else {
    stop("Token is neither an Oauth Token2.0 as created using make_new_token,
         nor a string for a personal access token")
  }
  return(invisible(NULL))
}
