% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sta.R
\name{sta}
\alias{sta}
\title{Statistical Seasonal Trend Analysis for numeric vector or \code{RasterStack}}
\usage{
sta(data, freq, numFreq = 4, delta = 0.2, startYear = 2000,
  endYear = 2018, intraAnnualPeriod = c("wetSeason", "drySeason"),
  interAnnualPeriod, adhocPeriod = NULL, significance = NULL,
  save = FALSE, dirToSaveSTA = NULL, numCores = 20)
}
\arguments{
\item{data}{numeric vector, matrix or RasterStack object}

\item{freq}{integer with the number of observations per period. See \code{Details}}

\item{numFreq}{integer with the number of frequencies to employ in harmonic regression fitting. 
See \code{\link[geoTS]{haRmonics}}}

\item{delta}{numeric (positive) controlling regularization and prevent non-invertible
hat matrix in harmonic regression model. See \code{\link[geoTS]{haRmonics}}}

\item{startYear}{numeric, time series initial year}

\item{endYear}{numeric, time series last year}

\item{intraAnnualPeriod}{character indicating seasons (wet or dry) to be considered for additional 
statistical analysis. See \code{Details}}

\item{interAnnualPeriod}{numeric vector indicating the number of years to be considered in STA. For instance,
1:5, indicates that the first five years will be
utilized for STA. Similarly, c(2,6,10) indicates that the second, sixth and tenth
years will be utilized for STA. See \code{Details}}

\item{adhocPeriod}{numeric vector with the specific observations to be considered in additional
statistical analysis. See \code{Details}}

\item{significance}{numeric in the interval (0,1) to assess statistical significance of trend analysis.
\code{NULL} by default.}

\item{save}{logical, should STA output be saved, default is \code{FALSE}}

\item{dirToSaveSTA}{character with full path name used to save \code{sta} progress report. When 
\code{save = TRUE}, \code{sta}'s output is saved on this path.}

\item{numCores}{integer given the number of cores to use; pertinent when \code{data} is a
RasterStack or a \code{\link[base]{matrix}}}
}
\value{
When \code{class(data)} is a \code{numeric}, an object of class "staNumeric" containing:
\item{data}{numeric vector}
\item{freq}{integer with the number of observations per period}
\item{startYear}{numeric, time series initial year}
\item{endYear}{numeric, time series last year}
\item{intraAnnualPeriod}{character indicating seasons (wet or dry)}
\item{interAnnualPeriod}{numeric vector indicating the number of years considered in STA}
\item{ts}{time series object; \code{data} in \code{\link[stats]{ts}} format}
\item{fit}{numeric vector with output of \code{\link[geoTS]{haRmonics}}}
\item{sta}{a list containing the following elements:}
\item{   }{\code{mean} a list of 12 elements with STA output for shape parameter \emph{mean}}
\item{   }{\code{annual} a list of 12 elements with STA output for shape parameter \emph{annual}}
\item{   }{\code{semiannual} a list of 12 elements with STA output for shape parameter \emph{semiannual}}
\item{significance}{numeric in the interval (0,1) or \code{NULL} when default used}

When \code{class(data)} is a \code{RasterStack} or a \code{\link[base]{matrix}}, an object of class
"staMatrix" containing:
\item{freq}{integer with the number of observations per period}
\item{daysUsedFit}{integer vector indicating the indices used to fit harmonic regression. see \code{\link[geoTS]{haRmonics}}}
\item{intervalsUsedBasicStats}{numeric vector indicating the indices used on calculation of basic statistics}
\item{sta}{a list containg the following elements:}
\item{   }{\code{mean} a matrix of 4 columns with STA output for shape parameter \emph{mean}. First two columns provide geolocation of analyzed pixels, 
third and fourth columns show p-value and slope estimate of STA, respectively}
\item{   }{\code{mean_basicStats} a matrix of 10 columns with basic statistics for shape parameter \emph{mean}. First two columns provide geolocation of analyzed pixels, 
from third to tenth columns show mean, standard deviation, global minimum, and maximum of minimum values, as well as mean, standard deviation,
global minimum, and maximum of maximum values, respectively}
\item{   }{\code{annual} a matrix of 4 columns with STA output for shape parameter \emph{annual}. First two columns provide geolocation of analyzed pixels, 
third and fourth columns show p-value and slope estimate of STA, respectively}
\item{   }{\code{annual_basicStats} a matrix of 10 columns with basic statistics for shape parameter \emph{annual}. First two columns provide geolocation of analyzed pixels, 
from third to tenth columns show mean, standard deviation, global minimum, and maximum of minimum values, as well as mean, standard deviation,
global minimum, and maximum of maximum values, respectively}
\item{   }{\code{semiannual} a matrix of 4 columns with STA output for shape parameter \emph{semiannual}. First two columns provide geolocation of analyzed pixels, 
third and fourth columns show p-value and slope estimate of STA, respectively}
\item{   }{\code{semiannual_basicStats} a matrix of 10 columns with basic statistics for shape parameter \emph{semiannual}. First two columns provide geolocation of analyzed pixels, 
from third to tenth columns show mean, standard deviation, global minimum, and maximum of minimum values, as well as mean, standard deviation,
global minimum, and maximum of maximum values, respectively}
}
\description{
Statistical Seasonal Trend Analysis for numeric vector or \code{RasterStack}
}
\details{
When the input is a \code{\link[base]{matrix}}, its first two columns must correspond 
to geographic coordinates. For instance, the matrix resulting from applying \code{\link[raster]{rasterToPoints}} 
to a \code{RasterStack} has this format. 

\code{freq} must be either 12 (monthly observations), 23 (Landsat annual scale) or 36 (10-day composite)
as this version implements STA for time series with these frequencies.

This version sets \code{intraAnnualPeriod} to either the \code{wetSeason} or the \code{drySeason} 
of Mexico. Empirical evidence suggests that while wet season runs from May to October, dry season
runs from November to April. Should a desired STA require specific months/days, these must be
provided through \code{adhocPeriod}.

When \code{interAnnualPeriod} is not specified and \code{class(data)=numeric},
\code{interAnnualPeriod = 1:(length(data)/freq)}; when \code{class(data)} is either \code{RasterStack}
or \code{\link[base]{matrix}}, \code{interAnnualPeriod = 1:((ncol(data)-2)/freq)}.

Since \code{adhocPeriod} defines an inter annual period "ad-hoc", the specific days of this ad-hoc
season must be known in advance and consequently, the specific time-points (with respect to the 
time series under consideration) must be provided in a numeric vector.

When \code{save=T}, a valid \code{dirToSaveSTA} must be provided, that is, this folder should have been
created previously. In this case, \code{sta}'s output is saved on \code{dirToSaveSTA}. This version
saves arrays of STA of the mean, annual and semi-annual parameters (along with their corresponding basic statistics)
in the file \code{sta_matrix_output.RData} inside \code{dirToSaveSTA}. Also, in the same directory,
the file \code{sta_progress.txt} records the progress of the STA process.

\code{save=T}, \code{dirToSaveSTA}, \code{numCores} and \code{master} are required when \code{data} is either a
\code{RasterStack} or a \code{\link[base]{matrix}}. The aforementioned basic statistics are: mean and standard deviation
of the time series of annual maximum and minimum as well as the global minima and maxima.
}
\note{
STA is based on the following ideas. Let \eqn{y_t} denote the value of a periodic time
series at time-point \eqn{t}. It is well-known that this type of observations can be modeled
as:

\eqn{y_t = a_0 + a_1 cos( (2 \pi t)/L - \phi_1) + ... +  a_K cos( (2 \pi K t)/L - \phi_K) + \varepsilon_t}, \eqn{t=1,\ldots,L}.
                
This model is known as harmonic regression. \eqn{L} denotes the number of observations per period, \eqn{K} is the number of 
harmonics included in the fit, \eqn{a_i}'s and \eqn{\phi_i}'s are called amplitude coefficients and phase angles,
respectively. \eqn{K} can be known empirically. Amplitudes and phase angle can be obtained as the solution of a least-squares
problem. 

In vegetation monitoring, amplitudes and phase angles are known as \emph{shape parameters}. In particular,
\eqn{a_0}, \eqn{a_1} and \eqn{a_2} are called \emph{mean} and \emph{annual} and \emph{semiannual} amplitudes, respectively.
Applying the harmonic regression model to observations over \eqn{P} periods of length \eqn{L} each, results
in estimates of shape parameters for each period. Thus, focusing only on amplitudes, \code{\link{sta}} yields
time series of mean, annual and semiannual parameters. A subsequent Mann-Kendall test for trend is performed
on each of these series.
}
\examples{
sta_marismas <- sta(data=marismas, freq=36)
str(sta_marismas)
plot(sta_marismas)
plot(sta_marismas, significance=0.09)

# Use of interAnnualPeriod
sta_21016 <- sta(data = marismas, freq = 36, interAnnualPeriod = c(2, 10, 16))
plot(sta_21016)

# Use of intraAnnualPeriod
sta_drySeason_218 <- sta(data = marismas, freq = 36,
                     interAnnualPeriod = 2:18, intraAnnualPeriod = "drySeason")
plot(sta_drySeason_218)

# Use of adhocPeriod and significance
adhoc <- list()
beginPeriod <- (1:17) * 36
endPeriod <- 2:18 * 36 
adhoc$partial <- c( sapply(1:length(beginPeriod), 
                 function(s) c(beginPeriod[s]+1, endPeriod[s]) ) )
adhoc$full <- c( sapply(1:length(beginPeriod), 
              function(s) (beginPeriod[s]+1):endPeriod[s]) )
sta_adhoc_218 <- sta(data = marismas, freq = 36, interAnnualPeriod = 2:18,
                 startYear = 2000, endYear = 2018, adhocPeriod = adhoc, significance=0.05)
plot(sta_adhoc_218)

# Use of ndmi RasterStack
\donttest{
ndmi_path = system.file("extdata", "ndmi.tif", package = "sta")
ndmiSTACK <- stack(ndmi_path)
dir.create(path=paste0(system.file("extdata", package="sta"), "/output_ndmi"),
          showWarnings=F)
outputDIR = paste0(system.file("extdata", package="sta"), "/output_ndmi")

sta_ndmi_21016 <- sta(data = ndmiSTACK, freq = 36,
                  numFreq = 4, delta = 0.2, intraAnnualPeriod = "wetSeason",
                  startYear = 2000, endYear = 2018, interAnnualPeriod = c(2,10,16),
                  save = TRUE, numCores = 5, dirToSaveSTA = outputDIR)}
 
}
\references{
Eastman, R., Sangermano, F., Ghimine, B., Zhu, H., Chen, H., Neeti, N., Cai, Y., Machado E., Crema, S. (2009).
\emph{Seasonal trend analysis of image time series},
International Journal of Remote Sensing, \bold{30(10)}, 2721--2726.
}
