% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kin.R
\name{kin.search}
\alias{kin.search}
\title{Midline tracking over image sequences}
\usage{
kin.search(
  image.dir = NULL,
  frames = NULL,
  thr = "otsu",
  plot.pml = TRUE,
  show.prog = FALSE,
  ant.per = 0.1,
  tips = 0.02,
  smoothing = "loess",
  smooth = 0.25,
  smooth.points = 200,
  image.type = "orig",
  save = TRUE,
  out.dir = NULL,
  flip = TRUE,
  size.min = 0.02,
  search.for = "largest",
  edges = FALSE,
  border = 5
)
}
\arguments{
\item{image.dir}{character, directory containing images to analyze.}

\item{frames}{numeric, vector indicating which images to process.}

\item{thr}{numeric or character ('otsu') threshold to determine binary image. See Details.}

\item{plot.pml}{logical, value indicating if outputted images should include an overlay of the theoretical midline based on \code{ant.per}.}

\item{show.prog}{logical value indicating if outputted image should be displayed during analysis.}

\item{ant.per}{numeric; left-most percentage of ROI that establishes the horizontal reference for the midline displacement.}

\item{tips, }{numeric, the proportion the the midline data to use in calculation of the head and tail position.}

\item{smoothing}{character, the midline smoothing method, either 'loess' or "spline".}

\item{smooth}{numeric; if \code{smoothing} is set to 'loess', smoothing parameter value for plotted midline.}

\item{smooth.points}{numeric, number of equally spaced points along the ROI midline on which the smoothed midline is computed.}

\item{image.type}{character; the type of image to be outputted, either 'orig' or 'bin' representing the original or binary images, respectively. Ignored if 'save=FALSE'.}

\item{save}{logical, value indicating if images should be outputted with midline and predicted midline based on the \code{ant.per} \code{lm()} overlaying original or binary images.}

\item{out.dir}{character, the directory to which outputted images should be saved.}

\item{flip}{logical, indicating if binary should be flipped.}

\item{size.min}{numeric, indicating the minimum size of ROIs as a proportion of the pixel field to be considered in analysis. May be useful if smaller unimportant ROIs appear in the frame. Default is 0.02.}

\item{search.for}{character, the search parameter. See Details.}

\item{edges}{logical, should ROIs on image edges be evaluated. See Details.}

\item{border}{if \code{edges=TRUE}, size of border to add in pixels. Dee details.}
}
\value{
A list with the following components:	

\code{kin.dat} a data table consisting of frame-by-frame position parameters for the ROI determined by \code{search.for}.	
\itemize{	
\item the frame number	
\item 'x' and ''y': the position of the tail (rightmost or posteriormost)	
\item 'head.x' and 'head.y': the x and y position of the head (leftmost or anteriormost)	
\item 'amp': the amplitude (\code{amp}) of the tail relative to thr theoretical midline determined by the \code{lm()} predictions from \code{ant.per}	
\item 'head.pval': p values of the \code{lm()} fit that describes the position of the head as determined by \code{ant.per} (green points in the outputted images/video)	
\item 'roi': a character indicating the ROI ranked by size ('a' being the largest)	
\item 'edge': indicating whether ROI was on the edge of the image field	
\item 'size': size of the ROI in pixels^2	
\item 'offset.x': ROI distance from horizontal center	
\item 'offset.y': ROI distance from vertical center	
\item 'offset': linear distance of ROI's centroid to image center	
}	

\code{midline} A data table containing, for each frame described by \code{frames}, the following: 	
\itemize{	
\item 'x' and 'y.m': x and y positions of the midline of the ROI	
#' \item 'y.min' and 'y.max': min and max y positions ROI's contour used in y.m calculation	
\item 'mid.pred': the predicted linear midline based on the points/pixels defined by \code{head.per} (green points in the outputted images/video)	
\item 'y.pred': midline points fit to a smooth spline or loess model with spar or span equal to \code{smooth} (red curve in the outputted images/video)	
\item 'wave.y': midline points 'y.pred' relative to 'mid.pred'	
\item 'roi': a character indicating ROI size ('a' being the largest)	
}	
	
\code{cont} A data table containing x and y positions of the contours used to calculate the data in 'kin.dat'. Contains the following: 	
\itemize{	
\item 'frame': the frame	
\item 'x' and 'y': the x and y positions of the contours	
}	
	
\code{all.classes} A data table containing the following for all ROIs detected:	
\itemize{	
\item 'frame': the frame	
\item 'roi': the name of each ROI found in a frame.	
\item 'edge': indicating whether ROI was on the edge of the image field	
\item 'size': size of the ROI in pixels^2	
\item 'offset.x': ROI distance from horizontal center	
\item 'offset.y': ROI distance from vertical center	
\item 'offset': linear distance of ROI's centroid to image center	
}	
	
\code{dim} the x and y dimensions of the images analyzed

A list with the following components:	

\code{kin.dat} a data frame consisting of frame-by-frame position parameters for the ROI indicated by \code{n.blob}:	
\itemize{	
\item the frame number	

\item 'head.x' and 'head.y': the x and y position of the head (leftmost or anteriormost)	
\item 'x' and 'y': the position of the tail (rightmost or posteriormost)	
\item 'amp': the amplitude (\code{amp}) of the tail	
\item 'cent.x' and 'cent.y': centroid coordinate of ROI	
\item 'roi': a character indicating ROI size ('a' being the largest)	
\item 'head.pval': p values of the \code{lm()} fit that describes the position of the head as determined by \code{ant.per} (green points in the outputted images/video)}	

\code{midline} A data frame containing, for each frame described by \code{frames}, the following: \itemize{	
\item 'x' and 'y.m': x and y positions of the midline of the ROI	
\item 'mid.pred': the predicted linear midline based on the points/pixels defined by \code{head.per} (green points in the outputted images/video)	
\item 'y.pred': midline points fit to a smooth spline or loess model with spar or span equal to \code{smooth} (red curve in the outputted images/video)	
\item 'wave.y': midline points 'y.pred' normalized to 'mid.pred'	
\item 'roi': a character indicating ROI size ('a' being the largest)	
\item 'cent.x': x centroid of ROI	
\item 'cent.y': y centroid of ROI	
\item 'offset.x': ROI distance from horizontal center	
\item 'offset.y': ROI distance from vertical center	
\item 'offset.total': sum of ROI offset.x and offset.y	
\item 'ar': aspect ration of the ROI	
\item 'size': size of ROI in pixels	
}	
	
\code{dim} the x and y dimensions of the images analyzed
}
\description{
Automatically retrieves the midline of a detected ROI in each image of a sequence through thresholding and segmentation; finds the y-value midpoint along the x-value array of the ROI and fits a midline according to a chosen smoothing method (loess or spline). Also outputs the midline amplitude relative to a reference line determined by an anterior section of the ROI. Supported image formats are jpeg, png, and tiff.
}
\details{
The algorithm assumes a left-right orientation, i.e., the head of the ROI is positioned left, the tail right. The \code{ant.per} value therefor establishes the reference line (theoretical straight midline) based on that portion of the head. The midline is calculated as the midpoints between the y extrema for each x position. Chooses ROIs based on relative ROI size or position.	

Thresholding operations can be performed with an arbitrary (user defined) numeric value or with Otsu's method ('thr="otsu"'). The latter chooses a threshold value by minimizing the combined intra-class variance. See \code{\link{otsu}}.	

If 'edges=TRUE', it is best to add an artificial border so that any part of the ROI in contact with the edge can be distinguished from it.

\code{search.for} determines how ROIs are chosen: 	
\itemize{	
\item "offset", the ROI with a centroid that is the shortest linear distance to the center of the field	
\item "offset.x", the ROI with a centroid x position that is closest to the x position of the center of the field	
\item "offset.y", the ROI with a centroid y position that is closest to the y position of the center of the field	
\item "largest", the largest ROI. 	
}	
	
These choices will be made on ROI sets that are not on the edge of the field if 'edges=FALSE'.	
	
\code{edges} Set by default to 'FALSE'. It is not advisable to include shapes that are on the edge of any frame and are therefore incomplete.	Yet, if set to 'TRUE', the \code{border} adds a black border to the image so that the intended ROI may be distinguished from the edge.
	
\code{image.type} Can be set as "orig" or "bin". "orig" plots midline and reference lines over the original video frames, "bin" over binary images.
}
\examples{

#### plot lot caudal amplitude and produce a classic midline waveform plot of swimming fish	
##A very long example.
\dontrun{	

#download example images and place in 'example' subdirectory	
f <- "https://github.com/ckenaley/exampledata/blob/master/example.zip?raw=true"	

download.file(f, paste0(tempdir(),"/temp.zip"))
unzip(paste0(tempdir(),"/temp.zip"), exdir=tempdir())	
unlink(paste0(tempdir(),"/temp.zip"))	

dir.create(paste0(tempdir(),"/processed_images"))
kin <- kin.search(image.dir =paste0(tempdir(),"/example"),	
       search.for = "largest",	
      smoothing = "loess",frames=1:50,
      out.dir=paste0(tempdir(),"/processed_images"),
      show.prog = FALSE,thr = "otsu",	
      image.type="bin",smooth=0.4)	

#plot instantaneous amplitude of tail (last/rightmost point) over frames 	
p <- ggplot(dat=kin$kin.dat,aes(x=frame,y=amp))+geom_line()+geom_point()+theme_classic(15)
print(p)	
	
# midline plot	
ml <- kin$midline	
#leftmost x starts at 0	
ml <- ml[,x2:=x-x[1],by=frame]	

ml <- merge(ml,kin$kin.dat[,list(frame,amp)],by="frame") #merge these	

pal <- wes_palette("Zissou1", 100, type = "continuous") #"Zissou" color palette

p <- ggplot(dat=ml,aes(x=x2,y=wave.y))+theme_classic(15)+scale_color_gradientn(colours = pal)	
p <- p+geom_line(aes(group=frame,color=amp),stat="smooth",method = "loess", size = 1.5)	
print(p)	
	
#Make a video of processed frames	

images.to.video2(image.dir=paste0(tempdir(),"/processed_images"),	
vid.name="trout_test",out.dir=tempdir(),frame.rate=5,qual=100,raw=FALSE)	
file.exists(paste0(tempdir(),"/trout_test_red.mp4"))

}

## A very short example.

#retrieve image 
i <- EBImage::readImage(system.file("extdata/img", "sunfish_BCF.jpg", package = "trackter"))
#create directory and write image to it
t <- tempdir()


dir.create(paste0(t,"/images"))
EBImage::writeImage(i,paste0(t,"/images/sunfish001.jpg"),type = "jpeg")

list.files(paste0(t,"/images"))
#run kin.search and save output image to directory
kin.i<- kin.search(image.dir = paste0(t,"/images"),smooth=0.7,save = TRUE,out.dir = t)

#plot midline over original image
with(kin.i$midline,plot(x,wave.y))

i2 <- EBImage::readImage(paste0(t,"/sunfish001_000.jpg"))
EBImage::display(i2,method="raster")

#clean up
unlink(paste0(t,"/images"),recursive=TRUE)

}
\seealso{
\code{\link{kin.simple}}
}
