% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/data_transformation_functions.R
\name{select_cycles}
\alias{select_cycles}
\title{Select cycles from a work loop object}
\usage{
select_cycles(
  x,
  cycle_def,
  keep_cycles = 4:6,
  bworth_order = 2,
  bworth_freq = 0.05,
  ...
)
}
\arguments{
\item{x}{A \code{workloop} object (see Details for how it should be
organized)}

\item{cycle_def}{A string specifying how cycles should be defined; one of:
"lo", "p2p", or "t2t". See Details more info}

\item{keep_cycles}{The indices of the cycles to keep. Include 0 to keep data
identified as being outside complete cycles}

\item{bworth_order}{Filter order for low-pass filtering of \code{Position}
via \code{signal::butter()} prior to finding L0}

\item{bworth_freq}{Critical frequency (scalar) for low-pass filtering of
\code{Position} via \code{signal::butter()} prior to finding L0}

\item{...}{Additional arguments passed to/from other functions that make use
of \code{select_cycles()}}
}
\value{
A \code{workloop} object with rows subsetted by the chosen position
cycles. A \code{Cycle} column is appended to denote which cycle each time
point is associated with. Finally, all attributes from the input
\code{workloop} object are retained and one new attribute is added to
record which cycles from the original data were retained.
}
\description{
Retain data from a work loop experiment based on position cycle
}
\details{
\code{select_cycles()} subsets data from a workloop trial by
position cycle. The \code{cycle_def} argument is used to specify which part
of the cycle is understood as the beginning and end. There are currently
three options: \cr
'lo' for L0-to-L0; \cr
'p2p' for peak-to-peak; and \cr
't2t' for trough-to-trough \cr

Peaks are identified using \code{pracma::findpeaks()}. L0 points on the
rising edge are found by finding the midpoints between troughs and the
following peak. However the first and last extrema and L0 points may be
misidentified by this method. Please plot your \code{Position} cycles to
ensure the edge cases are identified correctly.

The \code{keep_cycles} argument is used to determine which cycles (as
defined by \code{cycle_def} should be retained in the final dataset. Zero
is the index assigned to all data points that are determined to be outside
a complete cycle.

The \code{muscle_stim} object (\code{x}) must be a \code{workloop},
preferably read in by one of our data import functions. Please see
documentation for \code{as_muscle_stim()} if you need to manually construct
a \code{muscle_stim} object from another source.
}
\examples{

library(workloopR)

# import the workloop.ddf file included in workloopR
wl_dat <-read_ddf(system.file("extdata", "workloop.ddf",
                              package = 'workloopR'),
                  phase_from_peak = TRUE)

# select cycles 3 through 5 via the peak-to-peak definition
wl_selected <- select_cycles(wl_dat, cycle_def = "p2p", keep_cycles = 3:5)


# are the cycles of (approximately) the same length?
summary(as.factor(wl_selected$Cycle))

}
\seealso{
\code{\link{analyze_workloop}},
\code{\link{read_analyze_wl}},
\code{\link{read_analyze_wl_dir}}

Other data transformations: 
\code{\link{fix_GR}()},
\code{\link{invert_position}()}

Other workloop functions: 
\code{\link{analyze_workloop}()},
\code{\link{fix_GR}()},
\code{\link{get_wl_metadata}()},
\code{\link{invert_position}()},
\code{\link{read_analyze_wl_dir}()},
\code{\link{read_analyze_wl}()},
\code{\link{summarize_wl_trials}()},
\code{\link{time_correct}()}
}
\author{
Vikram B. Baliga and Shreeram Senthivasan
}
\concept{data transformations}
\concept{workloop functions}
